/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.storage.transfermanager;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.api.core.BetaApi;
import com.google.api.core.ListenableFutureToApiFuture;
import com.google.api.gax.core.GaxProperties;
import com.google.api.gax.rpc.FixedHeaderProvider;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import com.google.cloud.storage.transfermanager.ChunkedDownloadCallable;
import com.google.cloud.storage.transfermanager.DirectDownloadCallable;
import com.google.cloud.storage.transfermanager.DownloadJob;
import com.google.cloud.storage.transfermanager.DownloadResult;
import com.google.cloud.storage.transfermanager.DownloadSegment;
import com.google.cloud.storage.transfermanager.ParallelDownloadConfig;
import com.google.cloud.storage.transfermanager.ParallelUploadConfig;
import com.google.cloud.storage.transfermanager.Qos;
import com.google.cloud.storage.transfermanager.TransferManager;
import com.google.cloud.storage.transfermanager.TransferManagerConfig;
import com.google.cloud.storage.transfermanager.TransferManagerUtils;
import com.google.cloud.storage.transfermanager.TransferStatus;
import com.google.cloud.storage.transfermanager.UploadCallable;
import com.google.cloud.storage.transfermanager.UploadJob;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.BinaryOperator;
import org.checkerframework.checker.nullness.qual.NonNull;

@BetaApi
final class TransferManagerImpl
implements TransferManager {
    private static final String USER_AGENT_ENTRY = "gcloud-tm/";
    private static final String LIBRARY_VERSION = GaxProperties.getLibraryVersion(TransferManagerConfig.class);
    private final TransferManagerConfig transferManagerConfig;
    private final ListeningExecutorService executor;
    private final Qos qos;
    private final Storage storage;

    TransferManagerImpl(TransferManagerConfig transferManagerConfig) {
        this.transferManagerConfig = transferManagerConfig;
        this.executor = MoreExecutors.listeningDecorator(Executors.newFixedThreadPool(transferManagerConfig.getMaxWorkers()));
        this.qos = transferManagerConfig.getQos();
        StorageOptions storageOptions = transferManagerConfig.getStorageOptions();
        String userAgent = storageOptions.getUserAgent();
        if (userAgent == null || !userAgent.contains(USER_AGENT_ENTRY)) {
            storageOptions = ((StorageOptions.Builder)storageOptions.toBuilder().setHeaderProvider(FixedHeaderProvider.create(ImmutableMap.of("User-Agent", USER_AGENT_ENTRY + LIBRARY_VERSION)))).build();
        }
        this.storage = (Storage)storageOptions.getService();
    }

    @Override
    public void close() throws Exception {
        this.executor.shutdownNow();
        this.executor.awaitTermination(5L, TimeUnit.MINUTES);
    }

    @Override
    @BetaApi
    public @NonNull UploadJob uploadFiles(List<Path> files, ParallelUploadConfig config) {
        Storage.BlobWriteOption[] opts = config.getWriteOptsPerRequest().toArray(new Storage.BlobWriteOption[0]);
        ArrayList uploadTasks = new ArrayList();
        for (Path file : files) {
            if (Files.isDirectory(file, new LinkOption[0])) {
                throw new IllegalStateException("Directories are not supported");
            }
            String blobName = TransferManagerUtils.createBlobName(config, file);
            BlobInfo blobInfo = BlobInfo.newBuilder(config.getBucketName(), blobName).build();
            UploadCallable callable = new UploadCallable(this.transferManagerConfig, this.storage, blobInfo, file, config, opts);
            uploadTasks.add(TransferManagerImpl.convert(this.executor.submit((Callable)callable)));
        }
        return UploadJob.newBuilder().setParallelUploadConfig(config).setUploadResults(ImmutableList.copyOf(uploadTasks)).build();
    }

    @Override
    @BetaApi
    public @NonNull DownloadJob downloadBlobs(List<BlobInfo> blobs, ParallelDownloadConfig config) {
        Storage.BlobSourceOption[] opts = config.getOptionsPerRequest().toArray(new Storage.BlobSourceOption[0]);
        ArrayList<ApiFuture<DownloadResult>> downloadTasks = new ArrayList<ApiFuture<DownloadResult>>();
        if (!this.transferManagerConfig.isAllowDivideAndConquer()) {
            for (BlobInfo blob : blobs) {
                DirectDownloadCallable callable = new DirectDownloadCallable(this.storage, blob, config, opts);
                downloadTasks.add(TransferManagerImpl.convert(this.executor.submit((Callable)callable)));
            }
        } else {
            for (BlobInfo blob : blobs) {
                BlobInfo validatedBlob = TransferManagerImpl.retrieveSizeAndGeneration(this.storage, blob, config.getBucketName());
                Path destPath = TransferManagerUtils.createDestPath(config, blob);
                if (validatedBlob != null && this.qos.divideAndConquer(validatedBlob.getSize())) {
                    DownloadResult optimisticResult = DownloadResult.newBuilder(validatedBlob, TransferStatus.SUCCESS).setOutputDestination(destPath).build();
                    List downloadSegmentTasks = TransferManagerImpl.computeRanges(validatedBlob.getSize(), this.transferManagerConfig.getPerWorkerBufferSize()).stream().map(r -> new ChunkedDownloadCallable(this.storage, validatedBlob, opts, destPath, ((Range)r).begin, ((Range)r).end)).map(this.executor::submit).map(TransferManagerImpl::convert).collect(ImmutableList.toImmutableList());
                    downloadTasks.add(ApiFutures.transform(ApiFutures.allAsList(downloadSegmentTasks), segments -> segments.stream().reduce(optimisticResult, DownloadSegment::reduce, BinaryOperator.minBy(DownloadResult.COMPARATOR)), MoreExecutors.directExecutor()));
                    continue;
                }
                DirectDownloadCallable callable = new DirectDownloadCallable(this.storage, blob, config, opts);
                downloadTasks.add(TransferManagerImpl.convert(this.executor.submit((Callable)callable)));
            }
        }
        return DownloadJob.newBuilder().setDownloadResults(downloadTasks).setParallelDownloadConfig(config).build();
    }

    private static <T> ApiFuture<T> convert(ListenableFuture<T> lf) {
        return new ListenableFutureToApiFuture<T>(lf);
    }

    private static BlobInfo retrieveSizeAndGeneration(Storage storage, BlobInfo blobInfo, String bucketName) {
        if (blobInfo.getGeneration() == null) {
            return storage.get(BlobId.of(bucketName, blobInfo.getName()));
        }
        if (blobInfo.getSize() == null) {
            return storage.get(BlobId.of(bucketName, blobInfo.getName(), blobInfo.getGeneration()));
        }
        return blobInfo;
    }

    private static ImmutableList<Range> computeRanges(long end, long segmentSize) {
        ImmutableList.Builder b = ImmutableList.builder();
        if (end <= segmentSize) {
            b.add(Range.of(0L, end));
        } else {
            for (long i = 0L; i < end; i += segmentSize) {
                b.add(Range.of(i, Math.min(i + segmentSize, end)));
            }
        }
        return b.build();
    }

    private static final class Range {
        private final long begin;
        private final long end;

        private Range(long begin, long end) {
            this.begin = begin;
            this.end = end;
        }

        public static Range of(long begin, long end) {
            return new Range(begin, end);
        }
    }
}

