/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.http.nio;

import java.io.EOFException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Set;
import java.util.function.Predicate;
import javax.net.ssl.SSLException;
import org.broadinstitute.http.nio.HttpFileSystemProviderSettings;
import org.broadinstitute.http.nio.OutOfRetriesException;
import org.broadinstitute.http.nio.UnexpectedHttpResponseException;
import org.broadinstitute.http.nio.utils.ExceptionCauseIterator;
import org.broadinstitute.http.nio.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryHandler {
    public static final Set<String> DEFALT_RETRYABLE_MESSAGES = Set.of("protocol error:");
    public static final Set<Class<? extends Exception>> DEFAULT_RETRYABLE_EXCEPTIONS = Set.of(SSLException.class, EOFException.class, SocketException.class, SocketTimeoutException.class, InterruptedIOException.class);
    public static final Set<Integer> DEFAULT_RETRYABLE_HTTP_CODES = Set.of(Integer.valueOf(408), Integer.valueOf(429), Integer.valueOf(500), Integer.valueOf(502), Integer.valueOf(503), Integer.valueOf(504));
    private static final Logger LOGGER = LoggerFactory.getLogger(RetryHandler.class);
    private final int maxRetries;
    private final Set<Integer> retryableHttpCodes;
    private final Set<String> retryableMessages;
    private final Set<Class<? extends Exception>> retryableExceptions;
    private final Predicate<Throwable> customRetryPredicate;
    private final URI uri;

    public int getMaxRetries() {
        return this.maxRetries;
    }

    public RetryHandler(HttpFileSystemProviderSettings.RetrySettings settings, URI uri) {
        this(settings.maxRetries(), settings.retryableHttpCodes(), settings.retryableExceptions(), settings.retryableMessages(), settings.retryPredicate(), uri);
    }

    public RetryHandler(int maxRetries, Collection<Integer> retryableHttpCodes, Collection<Class<? extends Exception>> retryableExceptions, Collection<String> retryableMessages, Predicate<Throwable> retryPredicate, URI uri) {
        Utils.validateArg(maxRetries >= 0, "retries must be >= 0, was " + maxRetries);
        this.maxRetries = maxRetries;
        this.retryableHttpCodes = Set.copyOf(Utils.nonNull(retryableHttpCodes, () -> "retryableHttpCodes"));
        this.retryableExceptions = Set.copyOf(Utils.nonNull(retryableExceptions, () -> "retryableExceptions"));
        this.retryableMessages = Set.copyOf(Utils.nonNull(retryableMessages, () -> "retryableMessages"));
        this.customRetryPredicate = Utils.nonNull(retryPredicate, () -> "retryPredicate");
        this.uri = Utils.nonNull(uri, () -> "uri");
    }

    public void runWithRetries(IORunnable toRun) throws IOException {
        this.runWithRetries(() -> {
            toRun.run();
            return null;
        });
    }

    public <T> T runWithRetries(IOSupplier<T> toRun) throws IOException {
        return this.runWithRetries(toRun, null);
    }

    private <T> T runWithRetries(IOSupplier<T> toRun, IOException previousError) throws IOException {
        Duration totalSleepTime = Duration.ZERO;
        int tries = previousError == null ? 0 : 1;
        IOException mostRecentFailureReason = null;
        while (tries <= this.maxRetries) {
            try {
                ++tries;
                return toRun.get();
            }
            catch (IOException ex) {
                mostRecentFailureReason = ex;
                if (!this.isRetryable(ex)) {
                    throw ex;
                }
                LOGGER.warn("Retrying connection to {} due to error: {}. \nThis will be retry #{}", this.uri, ex.getMessage(), tries);
                totalSleepTime = totalSleepTime.plus(RetryHandler.sleepBeforeNextAttempt(tries));
            }
        }
        throw new OutOfRetriesException(tries - 1, totalSleepTime, mostRecentFailureReason);
    }

    public <T> T tryOnceThenWithRetries(IOSupplier<T> runFirst, IOSupplier<T> thenRunAndRetry) throws IOException {
        try {
            return runFirst.get();
        }
        catch (IOException initialFailure) {
            if (this.isRetryable(initialFailure)) {
                return this.runWithRetries(thenRunAndRetry, initialFailure);
            }
            throw initialFailure;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Duration sleepBeforeNextAttempt(int attempt) {
        Instant sleepEnd;
        Duration delay = Duration.ofMillis(1L << Math.min(attempt, 7));
        Instant sleepStart = Instant.now();
        try {
            Thread.sleep(delay.toMillis());
        }
        catch (InterruptedException iex) {
            Thread.currentThread().interrupt();
        }
        finally {
            sleepEnd = Instant.now();
        }
        return Duration.between(sleepStart, sleepEnd);
    }

    public boolean isRetryable(Exception exs) {
        for (Throwable cause : new ExceptionCauseIterator(exs)) {
            UnexpectedHttpResponseException responseException;
            if (cause instanceof UnexpectedHttpResponseException && this.retryableHttpCodes.contains((responseException = (UnexpectedHttpResponseException)cause).getResponseCode())) {
                return true;
            }
            for (Class<? extends Exception> retryable : this.retryableExceptions) {
                if (!retryable.isInstance(cause)) continue;
                return true;
            }
            for (String message : this.retryableMessages) {
                String errorMessage = cause.getMessage();
                if (errorMessage == null || !errorMessage.contains(message)) continue;
                return true;
            }
            if (!this.customRetryPredicate.test(cause)) continue;
            return true;
        }
        return false;
    }

    @FunctionalInterface
    public static interface IORunnable {
        public void run() throws IOException;
    }

    @FunctionalInterface
    public static interface IOSupplier<T> {
        public T get() throws IOException;
    }
}

