/*
 * Decompiled with CFR 0.152.
 */
package picard.illumina;

import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import picard.illumina.BarcodeMetric;
import picard.illumina.DistanceMetric;
import picard.illumina.parser.ReadDescriptor;
import picard.illumina.parser.ReadStructure;
import picard.illumina.parser.ReadType;
import picard.util.BarcodeEditDistanceQuery;
import picard.util.IlluminaUtil;

public class BarcodeExtractor {
    private final Map<String, BarcodeMetric> metrics = new HashMap<String, BarcodeMetric>();
    private final BarcodeMetric noMatch;
    private final Set<ByteString> barcodesBytes;
    private final int maxNoCalls;
    private final int maxMismatches;
    private final int minMismatchDelta;
    private final int minimumBaseQuality;
    private final DistanceMetric distanceMode;
    private static final int INITIAL_LOOKUP_SIZE = 4096;
    private final ConcurrentHashMap<ByteString, BarcodeMatch> barcodeLookupMap = new ConcurrentHashMap(4096);

    public BarcodeExtractor(Map<String, BarcodeMetric> barcodeToMetrics, BarcodeMetric noMatchMetric, ReadStructure readStructure, int maxNoCalls, int maxMismatches, int minMismatchDelta, int minimumBaseQuality, DistanceMetric distanceMode) {
        this.maxNoCalls = maxNoCalls;
        this.maxMismatches = maxMismatches;
        this.minMismatchDelta = minMismatchDelta;
        this.minimumBaseQuality = minimumBaseQuality;
        this.distanceMode = distanceMode;
        String[] noMatchBarcode = BarcodeExtractor.generateNoMatchBarcode(readStructure);
        byte[][] perfectScores = new byte[readStructure.sampleBarcodes.length()][];
        int index = 0;
        for (ReadDescriptor readDescriptor : readStructure.descriptors) {
            if (readDescriptor.type != ReadType.Barcode) continue;
            perfectScores[index] = new byte[readDescriptor.length];
            Arrays.fill(perfectScores[index], (byte)60);
            ++index;
        }
        this.noMatch = new BarcodeMetric(null, null, IlluminaUtil.barcodeSeqsToString(noMatchBarcode), noMatchBarcode);
        HashSet<ByteString> barcodesBytes = new HashSet<ByteString>(barcodeToMetrics.size());
        for (BarcodeMetric metric : barcodeToMetrics.values()) {
            barcodesBytes.add(new ByteString(metric.barcodeBytes));
            this.metrics.put(metric.BARCODE_WITHOUT_DELIMITER, metric);
        }
        this.barcodesBytes = barcodesBytes;
        for (BarcodeMetric metric : barcodeToMetrics.values()) {
            BarcodeMatch match = this.calculateBarcodeMatch(metric.barcodeBytes, perfectScores, true);
            this.barcodeLookupMap.put(new ByteString(metric.barcodeBytes), match);
        }
        BarcodeMatch barcodeMatch = this.calculateBarcodeMatch(noMatchMetric.barcodeBytes, perfectScores, true);
        this.barcodeLookupMap.put(new ByteString(noMatchMetric.barcodeBytes), barcodeMatch);
    }

    public Map<String, BarcodeMetric> getMetrics() {
        return this.metrics;
    }

    public BarcodeMetric getNoMatchMetric() {
        return this.noMatch;
    }

    BarcodeMatch findBestBarcode(byte[][] readSubsequences, byte[][] qualityScores, boolean isInlineMatching) {
        boolean canUseLookupTable = BarcodeExtractor.areAllQualitiesAboveMinimum(qualityScores, this.minimumBaseQuality);
        if (canUseLookupTable) {
            ByteString barcodesAsString = new ByteString(readSubsequences);
            BarcodeMatch match = this.barcodeLookupMap.get(barcodesAsString);
            if (match == null) {
                match = this.calculateBarcodeMatch(readSubsequences, qualityScores, isInlineMatching);
            }
            if (match.isMatched()) {
                this.barcodeLookupMap.put(barcodesAsString, match);
            }
            return match;
        }
        return this.calculateBarcodeMatch(readSubsequences, qualityScores, isInlineMatching);
    }

    BarcodeMatch calculateBarcodeMatch(byte[][] readSubsequences, byte[][] qualityScores, boolean isInlineMatching) {
        String bestBarcode = null;
        BarcodeMatch match = new BarcodeMatch();
        int totalBarcodeReadBases = 0;
        int numNoCalls = 0;
        for (byte[] bc : readSubsequences) {
            totalBarcodeReadBases += bc.length;
            for (byte b : bc) {
                if (SequenceUtil.isNoCall(b)) {
                    ++numNoCalls;
                }
                if (!isInlineMatching || numNoCalls <= this.maxNoCalls) continue;
                match.mismatches = totalBarcodeReadBases;
                match.barcode = "";
                match.matched = false;
                return match;
            }
        }
        int numMismatchesInBestBarcode = totalBarcodeReadBases + 1;
        int numMismatchesInSecondBestBarcode = totalBarcodeReadBases + 1;
        for (ByteString barcodeBytes : this.barcodesBytes) {
            BarcodeEditDistanceQuery barcodeEditDistanceQuery = new BarcodeEditDistanceQuery(barcodeBytes.bytes, readSubsequences, qualityScores, this.minimumBaseQuality, Math.min(this.maxMismatches, numMismatchesInBestBarcode) + this.minMismatchDelta);
            int numMismatches = this.distanceMode.distance(barcodeEditDistanceQuery);
            if (numMismatches < numMismatchesInBestBarcode) {
                if (bestBarcode != null) {
                    numMismatchesInSecondBestBarcode = numMismatchesInBestBarcode;
                }
                numMismatchesInBestBarcode = numMismatches;
                bestBarcode = barcodeBytes.toString();
                continue;
            }
            if (numMismatches >= numMismatchesInSecondBestBarcode) continue;
            numMismatchesInSecondBestBarcode = numMismatches;
        }
        match.matched = bestBarcode != null && numNoCalls <= this.maxNoCalls && numMismatchesInBestBarcode <= this.maxMismatches && numMismatchesInSecondBestBarcode - numMismatchesInBestBarcode >= this.minMismatchDelta;
        match.mismatches = numMismatchesInBestBarcode;
        match.mismatchesToSecondBest = numMismatchesInSecondBestBarcode;
        if (match.matched) {
            match.barcode = bestBarcode;
        } else if (!isInlineMatching && numNoCalls + numMismatchesInBestBarcode < totalBarcodeReadBases && bestBarcode != null) {
            match.barcode = bestBarcode.toLowerCase();
        } else {
            match.mismatches = totalBarcodeReadBases;
            match.barcode = "";
        }
        return match;
    }

    static void updateMetrics(BarcodeMatch match, boolean passingFilter, Map<String, BarcodeMetric> metrics, BarcodeMetric noMatchBarcodeMetric) {
        if (match.matched) {
            BarcodeMetric matchMetric = metrics.get(match.barcode);
            ++matchMetric.READS;
            if (passingFilter) {
                ++matchMetric.PF_READS;
            }
            if (match.mismatches == 0) {
                ++matchMetric.PERFECT_MATCHES;
                if (passingFilter) {
                    ++matchMetric.PF_PERFECT_MATCHES;
                }
            } else if (match.mismatches == 1) {
                ++matchMetric.ONE_MISMATCH_MATCHES;
                if (passingFilter) {
                    ++matchMetric.PF_ONE_MISMATCH_MATCHES;
                }
            }
        } else {
            ++noMatchBarcodeMetric.READS;
            if (passingFilter) {
                ++noMatchBarcodeMetric.PF_READS;
            }
        }
    }

    private static boolean areAllQualitiesAboveMinimum(byte[][] qualityScores, int minimumBaseQuality) {
        if (qualityScores == null) {
            return true;
        }
        byte[][] byArray = qualityScores;
        int n = byArray.length;
        for (int i = 0; i < n; ++i) {
            byte[] qs;
            for (byte q : qs = byArray[i]) {
                if (q >= minimumBaseQuality) continue;
                return false;
            }
        }
        return true;
    }

    public int getMinimumBaseQuality() {
        return this.minimumBaseQuality;
    }

    public static String[] generateNoMatchBarcode(ReadStructure inputReadStructure) {
        String[] noMatchBarcode = new String[inputReadStructure.sampleBarcodes.length()];
        int index = 0;
        for (ReadDescriptor d : inputReadStructure.descriptors) {
            if (d.type != ReadType.Barcode) continue;
            noMatchBarcode[index++] = StringUtil.repeatCharNTimes('N', d.length);
        }
        return noMatchBarcode;
    }

    private static final class ByteString {
        private final byte[][] bytes;
        private final int hash;

        public ByteString(byte[][] bytes) {
            this.bytes = new byte[bytes.length][];
            System.arraycopy(bytes, 0, this.bytes, 0, bytes.length);
            int h2 = 0;
            byte[][] byArray = this.bytes;
            int n = byArray.length;
            for (int i = 0; i < n; ++i) {
                byte[] bs;
                for (byte b : bs = byArray[i]) {
                    h2 = 31 * h2 + b;
                }
            }
            this.hash = h2;
        }

        public final int hashCode() {
            return this.hash;
        }

        public boolean equals(Object obj) {
            try {
                ByteString that = (ByteString)obj;
                if (this.hash != that.hash) {
                    return false;
                }
                if (this.bytes.length != that.bytes.length) {
                    return false;
                }
                for (int i = 0; i < this.bytes.length; ++i) {
                    if (Arrays.equals(this.bytes[i], that.bytes[i])) continue;
                    return false;
                }
                return true;
            }
            catch (Exception e) {
                return false;
            }
        }

        public String toString() {
            StringBuilder barcodeBuilder = new StringBuilder();
            for (byte[] barcode : this.bytes) {
                barcodeBuilder.append(new String(barcode, StandardCharsets.UTF_8));
            }
            return barcodeBuilder.toString();
        }
    }

    public static class BarcodeMatch {
        private boolean matched;
        private String barcode;
        private int mismatches;
        private int mismatchesToSecondBest;

        public boolean isMatched() {
            return this.matched;
        }

        public String getBarcode() {
            return this.barcode;
        }

        public int getMismatches() {
            return this.mismatches;
        }

        public int getMismatchesToSecondBest() {
            return this.mismatchesToSecondBest;
        }
    }
}

