/*
 * Decompiled with CFR 0.152.
 */
package picard.sam.SamErrorMetric;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.reference.SamLocusAndReferenceIterator;
import htsjdk.samtools.util.AbstractRecordAndOffset;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Locus;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityUtil;
import htsjdk.samtools.util.SamLocusIterator;
import htsjdk.tribble.util.ParsingUtils;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.vcf.VCFFileReader;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.sam.SamErrorMetric.BaseCalculator;
import picard.sam.SamErrorMetric.BaseErrorAggregation;
import picard.sam.SamErrorMetric.ErrorMetric;
import picard.sam.SamErrorMetric.ErrorType;
import picard.sam.SamErrorMetric.ReadBaseStratification;

@CommandLineProgramProperties(summary="Program to collect error metrics on bases stratified in various ways.\n<p>Sequencing errors come in different 'flavors'. For example, some occur during sequencing while others happen during library construction, prior to the sequencing. They may be correlated with various aspect of the sequencing experiment: position in the read, base context, length of insert and so on.\n <p>This program collects two different kinds of error metrics (one which attempts to distinguish between pre- and post- sequencer errors, and on which doesn't) and a collation of 'stratifiers' each of which assigns bases into various bins. The stratifiers can be used together to generate a composite stratification. <p>For example:<p>The BASE_QUALITY stratifier will place bases in bins according to their declared base quality. The READ_ORDINALITY stratifier will place bases in one of two bins depending on whether their read is 'first' or 'second'. One could generate a composite stratifier BASE_QUALITY:READ_ORDINALITY which will do both stratifications as the same time. \n<p>The resulting metric file will be named according to a provided prefix and a suffix which is generated  automatically according to the error metric. The tool can collect multiple metrics in a single pass and there should be hardly any performance loss when specifying multiple metrics at the same time; the default includes a large collection of metrics. \n<p>To estimate the error rate the tool assumes that all differences from the reference are errors. For this to be a reasonable assumption the tool needs to know the sites at which the sample is actually polymorphic and a confidence interval where the user is relatively certain that the polymorphic sites are known and accurate. These two inputs are provided as a VCF and INTERVALS. The program will only process sites that are in the intersection of the interval lists in the INTERVALS argument as long as they are not polymorphic in the VCF.\n\n", oneLineSummary="Program to collect error metrics on bases stratified in various ways.", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CollectSamErrorMetrics
extends CommandLineProgram {
    private static final int MAX_DIRECTIVES = ReadBaseStratification.Stratifier.values().length + 1;
    private static final Log log = Log.getInstance(CollectSamErrorMetrics.class);
    @Argument(shortName="I", doc="Input SAM or BAM file.")
    public String INPUT;
    @Argument(shortName="O", doc="Base name for output files. Actual file names will be generated from the basename and suffixes from the ERROR and STRATIFIER by adding a '.' and then error_by_stratifier[_and_stratifier]* where 'error' is ERROR's extension, and 'stratifier' is STRATIFIER's suffix. For example, an ERROR_METRIC of ERROR:BASE_QUALITY:GC_CONTENT will produce an extension '.error_by_base_quality_and_gc'. The suffixes can be found in the documentation for ERROR_VALUE and SUFFIX_VALUE.")
    public File OUTPUT;
    @Argument(doc="Errors to collect in the form of \"ERROR(:STRATIFIER)*\". To see the values available for ERROR and STRATIFIER look at the documentation for the arguments ERROR_VALUE and STRATIFIER_VALUE.")
    public List<String> ERROR_METRICS = CollectionUtil.makeList("ERROR", "ERROR:BASE_QUALITY", "ERROR:INSERT_LENGTH", "ERROR:GC_CONTENT", "ERROR:READ_DIRECTION", "ERROR:PAIR_ORIENTATION", "ERROR:HOMOPOLYMER", "ERROR:BINNED_HOMOPOLYMER", "ERROR:CYCLE", "ERROR:READ_ORDINALITY", "ERROR:READ_ORDINALITY:CYCLE", "ERROR:READ_ORDINALITY:HOMOPOLYMER", "ERROR:READ_ORDINALITY:GC_CONTENT", "ERROR:READ_ORDINALITY:PRE_DINUC", "ERROR:MAPPING_QUALITY", "ERROR:READ_GROUP", "ERROR:MISMATCHES_IN_READ", "ERROR:ONE_BASE_PADDED_CONTEXT", "OVERLAPPING_ERROR", "OVERLAPPING_ERROR:BASE_QUALITY", "OVERLAPPING_ERROR:INSERT_LENGTH", "OVERLAPPING_ERROR:READ_ORDINALITY", "OVERLAPPING_ERROR:READ_ORDINALITY:CYCLE", "OVERLAPPING_ERROR:READ_ORDINALITY:HOMOPOLYMER", "OVERLAPPING_ERROR:READ_ORDINALITY:GC_CONTENT", "INDEL_ERROR");
    @Argument(doc="A fake argument used to show the options of ERROR (in ERROR_METRICS).", optional=true)
    public ErrorType ERROR_VALUE;
    @Argument(doc="A fake argument used to show the options of STRATIFIER (in ERROR_METRICS).", optional=true)
    public ReadBaseStratification.Stratifier STRATIFIER_VALUE;
    @Argument(shortName="V", doc="VCF of known variation for sample. program will skip over polymorphic sites in this VCF and avoid collecting data on these loci.", optional=true)
    public String VCF;
    @Argument(shortName="L", doc="Region(s) to limit analysis to. Supported formats are VCF or interval_list. Will *intersect* inputs if multiple are given. When this argument is supplied, the VCF provided must be *indexed*.", optional=true)
    public List<File> INTERVALS;
    @Argument(shortName="MQ", doc="Minimum mapping quality to include read.")
    public int MIN_MAPPING_Q = 20;
    @Argument(shortName="BQ", doc="Minimum base quality to include base.")
    public int MIN_BASE_Q = 20;
    @Argument(shortName="PE", doc="The prior error, in phred-scale (used for calculating empirical error rates).", optional=true)
    public int PRIOR_Q = 30;
    @Argument(shortName="MAX", doc="Maximum number of loci to process (or unlimited if 0).", optional=true)
    public long MAX_LOCI;
    @Argument(shortName="LH", doc="Shortest homopolymer which is considered long.  Used by the BINNED_HOMOPOLYMER stratifier.", optional=true)
    public int LONG_HOMOPOLYMER = 6;
    @Argument(shortName="LBS", doc="Size of location bins. Used by the FLOWCELL_X and FLOWCELL_Y stratifiers", optional=true)
    public int LOCATION_BIN_SIZE = 2500;
    @Argument(shortName="P", doc="The probability of selecting a locus for analysis (for downsampling).", optional=true)
    public double PROBABILITY = 1.0;
    @Argument(fullName="PROGRESS_STEP_INTERVAL", doc="The interval between which progress will be displayed.", optional=true)
    public int PROGRESS_STEP_INTERVAL = 100000;
    @Argument(fullName="INTERVAL_ITERATOR", doc="Iterate through the file assuming it consists of a pre-created subset interval of the full genome.  This enables fast processing of files with reads at disparate parts of the genome.  Requires that the provided VCF file is indexed. ", optional=true)
    public boolean INTERVAL_ITERATOR = false;
    @Argument(shortName="EXT", doc="Append the given file extension to all metric file names (ex. OUTPUT.insert_size_metrics.EXT). No extension by default.", optional=true)
    public String FILE_EXTENSION = "";
    private Random random;
    private Collection<BaseErrorAggregation> aggregatorList;
    private ProgressLogger progressLogger;
    private long nTotalLoci;
    private long nSkippedLoci;
    private long nProcessedLoci;
    private SAMSequenceDictionary sequenceDictionary;
    private VCFFileReader vcfFileReader;
    private PeekableIterator<VariantContext> vcfIterator;
    private final HashMap<SAMRecord, SamLocusIterator.LocusInfo> previouslySeenDeletions = new HashMap();
    private SamLocusIterator.LocusInfo currentLocus = null;

    @Override
    protected boolean requiresReference() {
        return true;
    }

    @Override
    protected String[] customCommandLineValidation() {
        String[] superValidation;
        ArrayList<Object> errors = new ArrayList<Object>();
        if (this.ERROR_VALUE != null) {
            errors.add("ERROR_VALUE is a fake argument that is only there to show what are the different Error aggregation options. Please use it within the ERROR_METRICS argument.");
        }
        if (this.STRATIFIER_VALUE != null) {
            errors.add("STRATIFIER_VALUE is a fake argument that is only there to show what are the different Stratification options. Please use it within the STRATIFIER_VALUE argument.");
        }
        if (this.MIN_MAPPING_Q < 0) {
            errors.add("MIN_MAPPING_Q must be non-negative. found value: " + this.MIN_MAPPING_Q);
        }
        if (this.MIN_BASE_Q < 0) {
            errors.add("MIN_BASE_Q must be non-negative. found value: " + this.MIN_BASE_Q);
        }
        if (this.PRIOR_Q < 0) {
            errors.add("PRIOR_Q must be 2 or more. found value: " + this.PRIOR_Q);
        }
        if (this.MAX_LOCI < 0L) {
            errors.add("MAX_LOCI must be non-negative. found value: " + this.MAX_LOCI);
        }
        if (this.LONG_HOMOPOLYMER < 0) {
            errors.add("LONG_HOMOPOLYMER must be non-negative. found value: " + this.LONG_HOMOPOLYMER);
        }
        if (this.PROBABILITY < 0.0 || this.PROBABILITY > 1.0) {
            errors.add("PROBABILITY must be between 0 and 1. found value: " + this.PROBABILITY);
        }
        if ((superValidation = super.customCommandLineValidation()) != null) {
            errors.addAll(Arrays.asList(superValidation));
        }
        if (!errors.isEmpty()) {
            return errors.toArray(new String[0]);
        }
        return null;
    }

    private void initializeVcfDataSource() throws IOException {
        if (this.VCF == null) {
            this.vcfIterator = new PeekableIterator(Collections.emptyIterator());
        } else if (this.INTERVAL_ITERATOR) {
            this.vcfFileReader = new VCFFileReader(IOUtil.getPath(this.VCF), true);
            if (!this.vcfFileReader.isQueryable()) {
                throw new PicardException("Cannot query VCF File!  VCF Files must be queryable!  Please index input VCF and re-run.");
            }
        } else {
            this.vcfIterator = new PeekableIterator(new VCFFileReader(IOUtil.getPath(this.VCF), true).iterator());
        }
    }

    private boolean checkLocusForVariantOverlap(SamLocusIterator.LocusInfo locusInfo) {
        boolean returnValue;
        if (this.INTERVAL_ITERATOR) {
            returnValue = CollectSamErrorMetrics.checkLocus(this.vcfFileReader, locusInfo);
            if (returnValue) {
                log.debug("Locus overlaps a known variant: " + String.valueOf(locusInfo));
            }
        } else {
            returnValue = CollectSamErrorMetrics.advanceIteratorAndCheckLocus(this.vcfIterator, locusInfo, this.sequenceDictionary);
            if (returnValue) {
                log.debug(String.format("Locus overlaps a known variant from VCF: %s -> %s", locusInfo.toString(), this.vcfIterator.peek().toStringWithoutGenotypes()));
            }
        }
        return returnValue;
    }

    private void closeVcfDataSource() {
        if (this.INTERVAL_ITERATOR) {
            this.vcfFileReader.close();
        } else if (this.vcfIterator != null) {
            this.vcfIterator.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int processData() {
        try (SamReader sam = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(IOUtil.getPath(this.INPUT));
             ReferenceSequenceFileWalker referenceSequenceFileWalker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);){
            this.initializeVcfDataSource();
            SamLocusAndReferenceIterator iterator = this.createSamLocusAndReferenceIterator(sam, referenceSequenceFileWalker);
            log.info("Really starting iteration now.");
            for (SamLocusAndReferenceIterator.SAMLocusAndReference info : iterator) {
                if (this.random.nextDouble() > this.PROBABILITY) continue;
                ++this.nTotalLoci;
                if (this.checkLocusForVariantOverlap(info.getLocus())) {
                    log.debug("Skipping overlapping locus.");
                    ++this.nSkippedLoci;
                    continue;
                }
                this.addLocusBases(this.aggregatorList, info);
                ++this.nProcessedLoci;
                this.progressLogger.record(info.getLocus().getSequenceName(), info.getLocus().getPosition());
                if (this.MAX_LOCI == 0L || this.nProcessedLoci < this.MAX_LOCI) continue;
                log.warn("Early stopping due to having processed MAX_LOCI loci.");
                break;
            }
        }
        catch (IOException e) {
            log.error(e, "A problem occurred:", e.getMessage());
            int n = 1;
            return n;
        }
        finally {
            if (this.VCF != null) {
                this.closeVcfDataSource();
            }
        }
        return 0;
    }

    private SamLocusAndReferenceIterator createSamLocusAndReferenceIterator(SamReader sam, ReferenceSequenceFileWalker referenceSequenceFileWalker) {
        this.sequenceDictionary = referenceSequenceFileWalker.getSequenceDictionary();
        if (sam.getFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new PicardException("Input BAM must be sorted by coordinate");
        }
        this.sequenceDictionary.assertSameDictionary(sam.getFileHeader().getSequenceDictionary());
        IntervalList regionOfInterest = this.getIntervals(this.sequenceDictionary);
        log.info("Getting SamLocusIterator");
        SamLocusIterator samLocusIterator = new SamLocusIterator(sam, regionOfInterest);
        samLocusIterator.setIncludeIndels(true);
        samLocusIterator.setEmitUncoveredLoci(false);
        samLocusIterator.setMappingQualityScoreCutoff(this.MIN_MAPPING_Q);
        samLocusIterator.setQualityScoreCutoff(this.MIN_BASE_Q);
        log.info("Using " + this.aggregatorList.size() + " aggregators.");
        this.aggregatorList.forEach(la -> IOUtil.assertFileIsWritable(new File(String.valueOf(this.OUTPUT) + la.getSuffix() + this.FILE_EXTENSION)));
        log.info("Starting iteration over loci");
        SamLocusAndReferenceIterator iterator = new SamLocusAndReferenceIterator(referenceSequenceFileWalker, samLocusIterator);
        iterator.hasNext();
        return iterator;
    }

    private void initializeAggregationState() {
        this.random = new Random(42L);
        this.aggregatorList = this.getAggregatorList();
        this.progressLogger = new ProgressLogger(log, this.PROGRESS_STEP_INTERVAL);
        this.nTotalLoci = 0L;
        this.nSkippedLoci = 0L;
        this.nProcessedLoci = 0L;
    }

    @Override
    protected int doWork() {
        this.initializeAggregationState();
        try {
            IOUtil.assertFileIsReadable(IOUtil.getPath(this.INPUT));
            if (this.VCF != null) {
                IOUtil.assertFileIsReadable(IOUtil.getPath(this.VCF));
            }
            IOUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
            IOUtil.assertFilesAreReadable(this.INTERVALS);
        }
        catch (IOException e) {
            log.error(e, "A problem occurred:", e.getMessage());
            return 1;
        }
        int returnValue = this.processData();
        if (returnValue == 0) {
            log.info("Iteration complete, generating metric files");
            this.aggregatorList.forEach(this::writeMetricsFileForAggregator);
            log.info(String.format("Examined %d loci, Processed %d loci, Skipped %d loci.\nComputation took %d seconds.", this.nTotalLoci, this.nProcessedLoci, this.nSkippedLoci, this.progressLogger.getElapsedSeconds()));
        }
        return returnValue;
    }

    private void writeMetricsFileForAggregator(BaseErrorAggregation locusAggregator) {
        MetricsFile file = this.getMetricsFile();
        ErrorMetric.setPriorError(QualityUtil.getErrorProbabilityFromPhredScore(this.PRIOR_Q));
        for (ErrorMetric metric : locusAggregator.getMetrics()) {
            metric.calculateDerivedFields();
            boolean isSimpleError = locusAggregator.getSuffix().startsWith("error");
            if (metric.TOTAL_BASES == 0L && isSimpleError) continue;
            file.addMetric(metric);
        }
        file.write(new File(String.valueOf(this.OUTPUT) + "." + locusAggregator.getSuffix() + this.FILE_EXTENSION));
    }

    private static boolean advanceIteratorAndCheckLocus(PeekableIterator<VariantContext> vcfIterator, Locus locus, SAMSequenceDictionary sequenceDictionary) {
        while (vcfIterator.hasNext() && (vcfIterator.peek().isFiltered() || CollectSamErrorMetrics.CompareVariantContextToLocus(sequenceDictionary, vcfIterator.peek(), locus) < 0)) {
            vcfIterator.next();
        }
        return vcfIterator.hasNext() && CollectSamErrorMetrics.CompareVariantContextToLocus(sequenceDictionary, vcfIterator.peek(), locus) == 0;
    }

    @VisibleForTesting
    protected boolean processDeletionLocus(SamLocusIterator.RecordAndOffset deletionRao, SamLocusIterator.LocusInfo locusInfo) {
        if (this.currentLocus == null) {
            this.currentLocus = locusInfo;
        } else if (!this.currentLocus.withinDistanceOf(locusInfo, 0)) {
            this.currentLocus = locusInfo;
            this.previouslySeenDeletions.entrySet().removeIf(entry -> !((SamLocusIterator.LocusInfo)entry.getValue()).withinDistanceOf(this.currentLocus, 1));
        }
        if (this.previouslySeenDeletions.containsKey(deletionRao.getRecord())) {
            this.previouslySeenDeletions.put(deletionRao.getRecord(), this.currentLocus);
            return true;
        }
        this.previouslySeenDeletions.put(deletionRao.getRecord(), this.currentLocus);
        return false;
    }

    private void addRecordAndOffset(Collection<BaseErrorAggregation> aggregatorList, SamLocusIterator.RecordAndOffset rao, SamLocusAndReferenceIterator.SAMLocusAndReference info) {
        if (rao.getAlignmentType() == AbstractRecordAndOffset.AlignmentType.Deletion && this.processDeletionLocus(rao, info.getLocus())) {
            return;
        }
        for (BaseErrorAggregation aggregation : aggregatorList) {
            int mappingQuality = rao.getRecord().getMappingQuality();
            if (mappingQuality < this.MIN_MAPPING_Q) continue;
            if (rao.getAlignmentType() == AbstractRecordAndOffset.AlignmentType.Deletion) {
                aggregation.addBase(rao, info);
                continue;
            }
            byte baseQuality = rao.getRecord().getBaseQualities()[rao.getOffset()];
            if (baseQuality < this.MIN_BASE_Q) continue;
            aggregation.addBase(rao, info);
        }
    }

    public static int CompareVariantContextToLocus(SAMSequenceDictionary dictionary, VariantContext variantContext, Locus locus) {
        int indexDiff = dictionary.getSequenceIndex(variantContext.getContig()) - locus.getSequenceIndex();
        if (indexDiff != 0) {
            return indexDiff < 0 ? Integer.MIN_VALUE : Integer.MAX_VALUE;
        }
        if (locus.getPosition() < variantContext.getStart()) {
            return variantContext.getStart() - locus.getPosition();
        }
        if (locus.getPosition() > variantContext.getEnd()) {
            return variantContext.getEnd() - locus.getPosition();
        }
        return 0;
    }

    private static boolean checkLocus(VCFFileReader vcfFileReader, SamLocusIterator.LocusInfo locusInfo) {
        boolean overlaps = false;
        if (locusInfo != null) {
            try (CloseableIterator<VariantContext> vcfIterator = vcfFileReader.query(locusInfo);){
                while (vcfIterator.hasNext()) {
                    if (((VariantContext)vcfIterator.next()).isFiltered()) continue;
                    overlaps = true;
                    break;
                }
            }
        }
        return overlaps;
    }

    private void addLocusBases(Collection<BaseErrorAggregation> aggregatorList, SamLocusAndReferenceIterator.SAMLocusAndReference info) {
        for (SamLocusIterator.RecordAndOffset rao : info.getRecordAndOffsets()) {
            this.addRecordAndOffset(aggregatorList, rao, info);
        }
        for (SamLocusIterator.RecordAndOffset deletionRao : info.getLocus().getDeletedInRecord()) {
            this.addRecordAndOffset(aggregatorList, deletionRao, info);
        }
        for (SamLocusIterator.RecordAndOffset insertionRao : info.getLocus().getInsertedInRecord()) {
            this.addRecordAndOffset(aggregatorList, insertionRao, info);
        }
    }

    private Collection<BaseErrorAggregation> getAggregatorList() {
        ArrayList<BaseErrorAggregation> aggregatorList = new ArrayList<BaseErrorAggregation>();
        HashSet<String> suffixes = new HashSet<String>();
        ReadBaseStratification.setLongHomopolymer(this.LONG_HOMOPOLYMER);
        ReadBaseStratification.setLocationBinSize(this.LOCATION_BIN_SIZE);
        for (String directive : this.ERROR_METRICS) {
            BaseErrorAggregation aggregator = CollectSamErrorMetrics.parseDirective(directive);
            aggregatorList.add(aggregator);
            if (suffixes.add(aggregator.getSuffix())) continue;
            throw new IllegalArgumentException(String.format("Duplicated suffix (%s) found in aggregator %s.", aggregator.getSuffix(), aggregator.getClass()));
        }
        return aggregatorList;
    }

    private IntervalList getIntervals(SAMSequenceDictionary sequenceDictionary) {
        IntervalList regionOfInterest = null;
        for (File intervalListFile : this.INTERVALS) {
            if (!intervalListFile.exists()) {
                throw new IllegalArgumentException("Input file " + String.valueOf(intervalListFile) + " doesn't seem to exist. ");
            }
            log.info("Reading IntervalList ", intervalListFile, ".");
            IntervalList temp = IntervalList.fromFile(intervalListFile);
            sequenceDictionary.assertSameDictionary(temp.getHeader().getSequenceDictionary());
            if (regionOfInterest == null) {
                regionOfInterest = temp;
                continue;
            }
            log.info("Intersecting interval_list: ", intervalListFile, ".");
            regionOfInterest = IntervalList.intersection(regionOfInterest, temp);
        }
        return regionOfInterest;
    }

    protected static BaseErrorAggregation parseDirective(String stratifierDirective) {
        String[] directiveUnits = new String[MAX_DIRECTIVES + 1];
        int directiveSeparator = 58;
        int numberOfTerms = ParsingUtils.split(stratifierDirective, directiveUnits, ':', false);
        if (numberOfTerms > MAX_DIRECTIVES) {
            throw new IllegalArgumentException(String.format("Cannot parse more than the number of different stratifiers plus one (%d) terms in a single directive. (What are you trying to do?)", MAX_DIRECTIVES));
        }
        if (numberOfTerms == 0) {
            throw new IllegalArgumentException("Found no directives at all. Cannot process.");
        }
        List<ReadBaseStratification.RecordAndOffsetStratifier<?>> stratifiers = Arrays.stream(directiveUnits, 1, numberOfTerms).map(String::trim).map(ReadBaseStratification.Stratifier::valueOf).map(ReadBaseStratification.Stratifier::makeStratifier).collect(Collectors.toList());
        ReadBaseStratification.RecordAndOffsetStratifier<String> jointStratifier = stratifiers.isEmpty() ? ReadBaseStratification.nonStratifier : new ReadBaseStratification.CollectionStratifier(stratifiers);
        Supplier<? extends BaseCalculator> supplier = ErrorType.valueOf(directiveUnits[0].trim()).getErrorSupplier();
        return new BaseErrorAggregation<BaseCalculator>(supplier, jointStratifier);
    }
}

