/*
 * Decompiled with CFR 0.152.
 */
package picard.sam.markduplicates;

import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.SortingCollection;
import htsjdk.samtools.util.StringUtil;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.sam.DuplicationMetrics;
import picard.sam.DuplicationMetricsFactory;
import picard.sam.markduplicates.ElcDuplicatesFinderResolver;
import picard.sam.markduplicates.util.AbstractOpticalDuplicateFinderCommandLineProgram;
import picard.sam.util.PhysicalLocationShort;

@CommandLineProgramProperties(summary="Estimates the numbers of unique molecules in a sequencing library.  <p>This tool outputs quality metrics for a sequencing library preparation.Library complexity refers to the number of unique DNA fragments present in a given library.  Reductions in complexity resulting from PCR amplification during library preparation will ultimately compromise downstream analyses via an elevation in the number of duplicate reads.  PCR-associated duplication artifacts can result from: inadequate amounts of starting material (genomic DNA, cDNA, etc.), losses during cleanups, and size selection issues.  Duplicate reads can also arise from optical duplicates resulting from sequencing-machine optical sensor artifacts.</p>  <p>This tool attempts to estimate library complexity from sequence of read pairs alone.  Reads are sorted by the first N bases (5 by default) of the first read and then the first N bases of the second read of a pair.   Read pairs are considered to be duplicates if they match each other with no gaps and an overall mismatch rate less than or equal to MAX_DIFF_RATE (0.03 by default).  Reads of poor quality are filtered out to provide a more accurate estimate.  The filtering removes reads with any poor quality bases as defined by a read's MIN_MEAN_QUALITY (20 is the default value) across either the first or second read.  Unpaired reads are ignored in this computation.</p> <p>The algorithm attempts to detect optical duplicates separately from PCR duplicates and excludes these in the calculation of library size.  Also, since there is no alignment information used in this algorithm, an additional filter is applied to the data as follows.  After examining all reads, a histogram is built in which the number of reads in a duplicate set is compared with the number of of duplicate sets.   All bins that contain exactly one duplicate set are then removed from the histogram as outliers prior to the library size estimation.  </p><h4>Usage example:</h4><pre>java -jar picard.jar EstimateLibraryComplexity \\<br />     I=input.bam \\<br />     O=est_lib_complex_metrics.txt</pre>Please see the documentation for the companion <a href='https://broadinstitute.github.io/picard/command-line-overview.html#MarkDuplicates'>MarkDuplicates</a> tool.<hr />", oneLineSummary="Estimates the numbers of unique molecules in a sequencing library.  ", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class EstimateLibraryComplexity
extends AbstractOpticalDuplicateFinderCommandLineProgram {
    static final String USAGE_SUMMARY = "Estimates the numbers of unique molecules in a sequencing library.  ";
    static final String USAGE_DETAILS = "<p>This tool outputs quality metrics for a sequencing library preparation.Library complexity refers to the number of unique DNA fragments present in a given library.  Reductions in complexity resulting from PCR amplification during library preparation will ultimately compromise downstream analyses via an elevation in the number of duplicate reads.  PCR-associated duplication artifacts can result from: inadequate amounts of starting material (genomic DNA, cDNA, etc.), losses during cleanups, and size selection issues.  Duplicate reads can also arise from optical duplicates resulting from sequencing-machine optical sensor artifacts.</p>  <p>This tool attempts to estimate library complexity from sequence of read pairs alone.  Reads are sorted by the first N bases (5 by default) of the first read and then the first N bases of the second read of a pair.   Read pairs are considered to be duplicates if they match each other with no gaps and an overall mismatch rate less than or equal to MAX_DIFF_RATE (0.03 by default).  Reads of poor quality are filtered out to provide a more accurate estimate.  The filtering removes reads with any poor quality bases as defined by a read's MIN_MEAN_QUALITY (20 is the default value) across either the first or second read.  Unpaired reads are ignored in this computation.</p> <p>The algorithm attempts to detect optical duplicates separately from PCR duplicates and excludes these in the calculation of library size.  Also, since there is no alignment information used in this algorithm, an additional filter is applied to the data as follows.  After examining all reads, a histogram is built in which the number of reads in a duplicate set is compared with the number of of duplicate sets.   All bins that contain exactly one duplicate set are then removed from the histogram as outliers prior to the library size estimation.  </p><h4>Usage example:</h4><pre>java -jar picard.jar EstimateLibraryComplexity \\<br />     I=input.bam \\<br />     O=est_lib_complex_metrics.txt</pre>Please see the documentation for the companion <a href='https://broadinstitute.github.io/picard/command-line-overview.html#MarkDuplicates'>MarkDuplicates</a> tool.<hr />";
    @Argument(shortName="I", doc="One or more files to combine and estimate library complexity from. Reads can be mapped or unmapped.")
    public List<File> INPUT;
    @Argument(shortName="O", doc="Output file to writes per-library metrics to.")
    public File OUTPUT;
    @Argument(doc="The minimum number of bases at the starts of reads that must be identical for reads to be grouped together for duplicate detection.  In effect total_reads / 4^max_id_bases reads will be compared at a time, so lower numbers will produce more accurate results but consume exponentially more memory and CPU.")
    public int MIN_IDENTICAL_BASES = 5;
    @Argument(doc="The maximum rate of differences between two reads to call them identical.")
    public double MAX_DIFF_RATE = 0.03;
    @Argument(doc="The minimum mean quality of the bases in a read pair for the read to be analyzed. Reads with lower average quality are filtered out and not considered in any calculations.")
    public int MIN_MEAN_QUALITY = 20;
    @Argument(doc="Do not process self-similar groups that are this many times over the mean expected group size. I.e. if the input contains 10m read pairs and MIN_IDENTICAL_BASES is set to 5, then the mean expected group size would be approximately 10 reads.")
    public int MAX_GROUP_RATIO = 500;
    @Argument(doc="Barcode SAM tag (ex. BC for 10X Genomics)", optional=true)
    public String BARCODE_TAG = null;
    @Argument(doc="Read one barcode SAM tag (ex. BX for 10X Genomics)", optional=true)
    public String READ_ONE_BARCODE_TAG = null;
    @Argument(doc="Read two barcode SAM tag (ex. BX for 10X Genomics)", optional=true)
    public String READ_TWO_BARCODE_TAG = null;
    @Argument(doc="The maximum number of bases to consider when comparing reads (0 means no maximum).", optional=true)
    public int MAX_READ_LENGTH = 0;
    @Argument(doc="Minimum number group count.  On a per-library basis, we count the number of groups of duplicates that have a particular size.  Omit from consideration any count that is less than this value.  For example, if we see only one group of duplicates with size 500, we omit it from the metric calculations if MIN_GROUP_COUNT is set to two.  Setting this to two may help remove technical artifacts from the library size calculation, for example, adapter dimers.", optional=true)
    public int MIN_GROUP_COUNT = 2;
    private final Log log = Log.getInstance(EstimateLibraryComplexity.class);

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errorMsgs = new ArrayList<String>();
        if (0 < this.MAX_READ_LENGTH && this.MAX_READ_LENGTH < this.MIN_IDENTICAL_BASES) {
            errorMsgs.add("MAX_READ_LENGTH must be greater than MIN_IDENTICAL_BASES");
        }
        if (this.MIN_IDENTICAL_BASES <= 0) {
            errorMsgs.add("MIN_IDENTICAL_BASES must be greater than 0");
        }
        return errorMsgs.isEmpty() ? super.customCommandLineValidation() : errorMsgs.toArray(new String[errorMsgs.size()]);
    }

    public int getBarcodeValue(SAMRecord record) {
        return EstimateLibraryComplexity.getReadBarcodeValue(record, this.BARCODE_TAG);
    }

    public static int getReadBarcodeValue(SAMRecord record, String tag) {
        if (null == tag) {
            return 0;
        }
        String attr = record.getStringAttribute(tag);
        if (null == attr) {
            return 0;
        }
        return attr.hashCode();
    }

    private int getReadOneBarcodeValue(SAMRecord record) {
        return EstimateLibraryComplexity.getReadBarcodeValue(record, this.READ_ONE_BARCODE_TAG);
    }

    private int getReadTwoBarcodeValue(SAMRecord record) {
        return EstimateLibraryComplexity.getReadBarcodeValue(record, this.READ_TWO_BARCODE_TAG);
    }

    public EstimateLibraryComplexity() {
        int sizeInBytes = null != this.BARCODE_TAG || null != this.READ_ONE_BARCODE_TAG || null != this.READ_TWO_BARCODE_TAG ? PairedReadSequenceWithBarcodes.getSizeInBytes() : PairedReadSequence.getSizeInBytes();
        this.MAX_RECORDS_IN_RAM = Math.min((int)(Runtime.getRuntime().maxMemory() / (long)sizeInBytes) / 2, 0x7FFFFFDF);
    }

    @Override
    protected int doWork() {
        for (File f : this.INPUT) {
            IOUtil.assertFileIsReadable(f);
        }
        this.log.info("Will store " + this.MAX_RECORDS_IN_RAM + " read pairs in memory before sorting.");
        ArrayList<SAMReadGroupRecord> readGroups = new ArrayList<SAMReadGroupRecord>();
        boolean useBarcodes = null != this.BARCODE_TAG || null != this.READ_ONE_BARCODE_TAG || null != this.READ_TWO_BARCODE_TAG;
        SortingCollection<PairedReadSequence> sorter = !useBarcodes ? SortingCollection.newInstance(PairedReadSequence.class, new PairedReadCodec(), new PairedReadComparator(), (int)this.MAX_RECORDS_IN_RAM, this.TMP_DIR) : SortingCollection.newInstance(PairedReadSequence.class, new PairedReadWithBarcodesCodec(), new PairedReadComparator(), (int)this.MAX_RECORDS_IN_RAM, this.TMP_DIR);
        ProgressLogger progress = new ProgressLogger(this.log, 1000000, "Read");
        for (File f : this.INPUT) {
            HashMap<String, PairedReadSequence> pendingByName = new HashMap<String, PairedReadSequence>();
            SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(f);
            readGroups.addAll(in.getFileHeader().getReadGroups());
            for (SAMRecord rec : in) {
                PairedReadSequenceWithBarcodes prsWithBarcodes;
                if (!rec.getReadPairedFlag() || !rec.getFirstOfPairFlag() && !rec.getSecondOfPairFlag() || rec.isSecondaryOrSupplementary()) continue;
                PairedReadSequence prs = (PairedReadSequence)pendingByName.remove(rec.getReadName());
                if (prs == null) {
                    SAMReadGroupRecord rg;
                    PairedReadSequence pairedReadSequence = prs = useBarcodes ? new PairedReadSequenceWithBarcodes() : new PairedReadSequence();
                    if (this.opticalDuplicateFinder.addLocationInformation(rec.getReadName(), prs) && (rg = rec.getReadGroup()) != null) {
                        prs.setReadGroup((short)readGroups.indexOf(rg));
                    }
                    pendingByName.put(rec.getReadName(), prs);
                }
                boolean passesQualityCheck = this.passesQualityCheck(rec.getReadBases(), rec.getBaseQualities(), this.MIN_IDENTICAL_BASES, this.MIN_MEAN_QUALITY);
                prs.qualityOk = prs.qualityOk && passesQualityCheck;
                byte[] bases = rec.getReadBases();
                if (rec.getReadNegativeStrandFlag()) {
                    SequenceUtil.reverseComplement(bases);
                }
                PairedReadSequenceWithBarcodes pairedReadSequenceWithBarcodes = prsWithBarcodes = useBarcodes ? (PairedReadSequenceWithBarcodes)prs : null;
                if (rec.getFirstOfPairFlag()) {
                    prs.read1 = bases;
                    if (useBarcodes) {
                        prsWithBarcodes.barcode = this.getBarcodeValue(rec);
                        prsWithBarcodes.readOneBarcode = this.getReadOneBarcodeValue(rec);
                    }
                } else {
                    prs.read2 = bases;
                    if (useBarcodes) {
                        prsWithBarcodes.readTwoBarcode = this.getReadTwoBarcodeValue(rec);
                    }
                }
                if (prs.read1 != null && prs.read2 != null && prs.qualityOk) {
                    sorter.add(prs);
                }
                progress.record(rec);
            }
            CloserUtil.close(in);
        }
        this.log.info(String.format("Finished reading - read %d records - moving on to scanning for duplicates.", progress.getCount()));
        PeekableIterator<PairedReadSequence> iterator = new PeekableIterator<PairedReadSequence>(sorter.iterator());
        HashMap duplicationHistosByLibrary = new HashMap();
        HashMap opticalHistosByLibrary = new HashMap();
        int groupsProcessed = 0;
        long lastLogTime = System.currentTimeMillis();
        int meanGroupSize = (int)Math.max(1L, progress.getCount() / 2L / (long)((int)Math.pow(4.0, this.MIN_IDENTICAL_BASES * 2)));
        ElcDuplicatesFinderResolver algorithmResolver = new ElcDuplicatesFinderResolver(this.MAX_DIFF_RATE, this.MAX_READ_LENGTH, this.MIN_IDENTICAL_BASES, useBarcodes, this.opticalDuplicateFinder);
        while (iterator.hasNext()) {
            List<PairedReadSequence> group = this.getNextGroup(iterator);
            if (group.size() > meanGroupSize * this.MAX_GROUP_RATIO) {
                PairedReadSequence prs = group.get(0);
                this.log.warn("Omitting group with over " + this.MAX_GROUP_RATIO + " times the expected mean number of read pairs. Mean=" + meanGroupSize + ", Actual=" + group.size() + ". Prefixes: " + StringUtil.bytesToString(prs.read1, 0, this.MIN_IDENTICAL_BASES) + " / " + StringUtil.bytesToString(prs.read2, 0, this.MIN_IDENTICAL_BASES));
                continue;
            }
            Map<String, List<PairedReadSequence>> sequencesByLibrary = this.splitByLibrary(group, readGroups);
            for (Map.Entry<String, List<PairedReadSequence>> entry : sequencesByLibrary.entrySet()) {
                String library = entry.getKey();
                List<PairedReadSequence> seqs = entry.getValue();
                Histogram duplicationHisto = (Histogram)duplicationHistosByLibrary.get(library);
                Histogram opticalHisto = (Histogram)opticalHistosByLibrary.get(library);
                if (duplicationHisto == null) {
                    duplicationHisto = new Histogram("duplication_group_count", library);
                    opticalHisto = new Histogram("duplication_group_count", "optical_duplicates");
                    duplicationHistosByLibrary.put(library, duplicationHisto);
                    opticalHistosByLibrary.put(library, opticalHisto);
                }
                algorithmResolver.resolveAndSearch(seqs, duplicationHisto, opticalHisto);
            }
            ++groupsProcessed;
            if (lastLogTime >= System.currentTimeMillis() - 60000L) continue;
            this.log.info("Processed " + groupsProcessed + " groups.");
            lastLogTime = System.currentTimeMillis();
        }
        iterator.close();
        sorter.cleanup();
        MetricsFile file = this.getMetricsFile();
        for (String library : duplicationHistosByLibrary.keySet()) {
            Histogram duplicationHisto = (Histogram)duplicationHistosByLibrary.get(library);
            Histogram opticalHisto = (Histogram)opticalHistosByLibrary.get(library);
            DuplicationMetrics metrics = DuplicationMetricsFactory.createMetrics();
            metrics.LIBRARY = library;
            for (Integer bin : duplicationHisto.keySet()) {
                double opticalDuplicates;
                double duplicateGroups = duplicationHisto.get(bin).getValue();
                double d = opticalDuplicates = opticalHisto.get(bin) == null ? 0.0 : opticalHisto.get(bin).getValue();
                if (!(duplicateGroups >= (double)this.MIN_GROUP_COUNT)) continue;
                metrics.READ_PAIRS_EXAMINED = (long)((double)metrics.READ_PAIRS_EXAMINED + (double)bin.intValue() * duplicateGroups);
                metrics.READ_PAIR_DUPLICATES = (long)((double)metrics.READ_PAIR_DUPLICATES + (double)(bin - 1) * duplicateGroups);
                metrics.READ_PAIR_OPTICAL_DUPLICATES = (long)((double)metrics.READ_PAIR_OPTICAL_DUPLICATES + opticalDuplicates);
            }
            metrics.calculateDerivedFields();
            file.addMetric(metrics);
            file.addHistogram(duplicationHisto);
        }
        file.write(this.OUTPUT);
        return 0;
    }

    List<PairedReadSequence> getNextGroup(PeekableIterator<PairedReadSequence> iterator) {
        ArrayList<PairedReadSequence> group = new ArrayList<PairedReadSequence>();
        PairedReadSequence first = iterator.next();
        group.add(first);
        block0: while (iterator.hasNext()) {
            PairedReadSequence next = iterator.peek();
            for (int i = 0; i < this.MIN_IDENTICAL_BASES; ++i) {
                if (first.read1[i] != next.read1[i] || first.read2[i] != next.read2[i]) break block0;
            }
            group.add(iterator.next());
        }
        return group;
    }

    Map<String, List<PairedReadSequence>> splitByLibrary(List<PairedReadSequence> input, List<SAMReadGroupRecord> rgs) {
        HashMap<String, List<PairedReadSequence>> out = new HashMap<String, List<PairedReadSequence>>();
        for (PairedReadSequence seq : input) {
            ArrayList<PairedReadSequence> librarySeqs;
            String library;
            if (seq.getReadGroup() != -1) {
                library = rgs.get(seq.getReadGroup()).getLibrary();
                if (library == null) {
                    library = "Unknown";
                }
            } else {
                library = "Unknown";
            }
            if ((librarySeqs = (ArrayList<PairedReadSequence>)out.get(library)) == null) {
                librarySeqs = new ArrayList<PairedReadSequence>();
                out.put(library, librarySeqs);
            }
            librarySeqs.add(seq);
        }
        return out;
    }

    boolean passesQualityCheck(byte[] bases, byte[] quals, int seedLength, int minQuality) {
        if (bases.length < seedLength) {
            return false;
        }
        for (int i = 0; i < seedLength; ++i) {
            if (!SequenceUtil.isNoCall(bases[i])) continue;
            return false;
        }
        int maxReadLength = this.MAX_READ_LENGTH <= 0 ? Integer.MAX_VALUE : this.MAX_READ_LENGTH;
        int readLength = Math.min(bases.length, maxReadLength);
        int total = 0;
        for (int i = 0; i < readLength; ++i) {
            total += quals[i];
        }
        return total / readLength >= minQuality;
    }

    static class PairedReadSequenceWithBarcodes
    extends PairedReadSequence {
        int barcode;
        int readOneBarcode;
        int readTwoBarcode;

        public PairedReadSequenceWithBarcodes() {
        }

        public PairedReadSequenceWithBarcodes(PairedReadSequence val) {
            if (null == val) {
                throw new PicardException("val was null");
            }
            this.readGroup = val.getReadGroup();
            this.tile = val.getTile();
            this.x = val.getX();
            this.y = val.getY();
            this.qualityOk = val.qualityOk;
            this.read1 = (byte[])val.read1.clone();
            this.read2 = (byte[])val.read2.clone();
            this.libraryId = val.getLibraryId();
        }

        public static int getSizeInBytes() {
            return PairedReadSequence.getSizeInBytes() + 12;
        }
    }

    static class PairedReadSequence
    extends PhysicalLocationShort {
        static final int NUMBER_BASES_IN_READ = 150;
        short readGroup = (short)-1;
        boolean qualityOk = true;
        byte[] read1;
        byte[] read2;
        short libraryId;
        int[] hashes1;
        int[] hashes2;

        PairedReadSequence() {
        }

        public static int getSizeInBytes() {
            return 471;
        }

        @Override
        public short getReadGroup() {
            return this.readGroup;
        }

        @Override
        public void setReadGroup(short readGroup) {
            this.readGroup = readGroup;
        }

        @Override
        public short getLibraryId() {
            return this.libraryId;
        }

        @Override
        public void setLibraryId(short libraryId) {
            this.libraryId = libraryId;
        }

        public static SortingCollection.Codec<PairedReadSequence> getCodec() {
            return new PairedReadCodec();
        }

        void initHashes(int numberOfHashes, int skippedBases, int minReadLength) {
            this.hashes1 = this.getHashes(this.read1, numberOfHashes, skippedBases, minReadLength);
            this.hashes2 = this.getHashes(this.read2, numberOfHashes, skippedBases, minReadLength);
        }

        private int[] getHashes(byte[] read, int numberOfHashes, int skippedBases, int minReadLength) {
            int[] hashValues = new int[numberOfHashes];
            for (int i = 0; i < numberOfHashes; ++i) {
                hashValues[i] = 1;
                for (int position = skippedBases + i; position < minReadLength; position += numberOfHashes) {
                    hashValues[i] = 31 * hashValues[i] + read[position];
                }
            }
            return hashValues;
        }
    }

    static class PairedReadCodec
    implements SortingCollection.Codec<PairedReadSequence> {
        protected DataOutputStream out;
        protected DataInputStream in;

        PairedReadCodec() {
        }

        @Override
        public void setOutputStream(OutputStream out) {
            this.out = new DataOutputStream(out);
        }

        @Override
        public void setInputStream(InputStream in) {
            this.in = new DataInputStream(in);
        }

        @Override
        public void encode(PairedReadSequence val) {
            try {
                this.out.writeShort(val.readGroup);
                this.out.writeShort(val.tile);
                this.out.writeShort(val.x);
                this.out.writeShort(val.y);
                this.out.writeInt(val.read1.length);
                this.out.write(val.read1);
                this.out.writeInt(val.read2.length);
                this.out.write(val.read2);
            }
            catch (IOException ioe) {
                throw new PicardException("Error write out read pair.", ioe);
            }
        }

        @Override
        public PairedReadSequence decode() {
            try {
                PairedReadSequence val = new PairedReadSequence();
                try {
                    val.readGroup = this.in.readShort();
                }
                catch (EOFException eof) {
                    return null;
                }
                val.tile = this.in.readShort();
                val.x = this.in.readShort();
                val.y = this.in.readShort();
                int length = this.in.readInt();
                val.read1 = new byte[length];
                if (this.in.read(val.read1) != length) {
                    throw new PicardException("Could not read " + length + " bytes from temporary file.");
                }
                length = this.in.readInt();
                val.read2 = new byte[length];
                if (this.in.read(val.read2) != length) {
                    throw new PicardException("Could not read " + length + " bytes from temporary file.");
                }
                return val;
            }
            catch (IOException ioe) {
                throw new PicardException("Exception reading read pair.", ioe);
            }
        }

        @Override
        public SortingCollection.Codec<PairedReadSequence> clone() {
            return new PairedReadCodec();
        }
    }

    private class PairedReadComparator
    implements Comparator<PairedReadSequence> {
        final int BASES;

        private PairedReadComparator() {
            this.BASES = EstimateLibraryComplexity.this.MIN_IDENTICAL_BASES;
        }

        @Override
        public int compare(PairedReadSequence lhs, PairedReadSequence rhs) {
            int retval;
            int i;
            for (i = 0; i < this.BASES; ++i) {
                retval = lhs.read1[i] - rhs.read1[i];
                if (retval == 0) continue;
                return retval;
            }
            for (i = 0; i < this.BASES; ++i) {
                retval = lhs.read2[i] - rhs.read2[i];
                if (retval == 0) continue;
                return retval;
            }
            return 0;
        }
    }

    static class PairedReadWithBarcodesCodec
    extends PairedReadCodec {
        PairedReadWithBarcodesCodec() {
        }

        @Override
        public void encode(PairedReadSequence val) {
            if (!(val instanceof PairedReadSequenceWithBarcodes)) {
                throw new PicardException("Val was not a PairedReadSequenceWithBarcodes");
            }
            PairedReadSequenceWithBarcodes data = (PairedReadSequenceWithBarcodes)val;
            super.encode(val);
            try {
                this.out.writeInt(data.barcode);
                this.out.writeInt(data.readOneBarcode);
                this.out.writeInt(data.readTwoBarcode);
            }
            catch (IOException ioe) {
                throw new PicardException("Error write out read pair.", ioe);
            }
        }

        @Override
        public PairedReadSequence decode() {
            try {
                PairedReadSequence parentVal = super.decode();
                if (null == parentVal) {
                    return null;
                }
                PairedReadSequenceWithBarcodes val = new PairedReadSequenceWithBarcodes(parentVal);
                val.barcode = this.in.readInt();
                val.readOneBarcode = this.in.readInt();
                val.readTwoBarcode = this.in.readInt();
                return val;
            }
            catch (IOException ioe) {
                throw new PicardException("Exception reading read pair.", ioe);
            }
        }

        @Override
        public SortingCollection.Codec<PairedReadSequence> clone() {
            return new PairedReadWithBarcodesCodec();
        }
    }
}

