/*
 * Decompiled with CFR 0.152.
 */
package picard.util;

import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.GenotypesContext;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.ArrayUtils;

public class LiftoverUtils {
    public static final String SWAPPED_ALLELES = "SwappedAlleles";
    public static final String REV_COMPED_ALLELES = "ReverseComplementedAlleles";
    public static final Collection<String> DEFAULT_TAGS_TO_REVERSE = Collections.singletonList("AF");
    public static final Collection<String> DEFAULT_TAGS_TO_DROP = Collections.singletonList("MAX_AF");
    public static final Log log = Log.getInstance(LiftoverUtils.class);

    public static VariantContext liftVariant(VariantContext source, Interval target, ReferenceSequence refSeq, boolean writeOriginalPosition, boolean writeOriginalAlleles) {
        if (target == null) {
            return null;
        }
        VariantContextBuilder builder = target.isNegativeStrand() ? LiftoverUtils.reverseComplementVariantContext(source, target, refSeq) : LiftoverUtils.liftSimpleVariantContext(source, target);
        if (builder == null) {
            return null;
        }
        builder.filters(source.getFilters());
        builder.log10PError(source.getLog10PError());
        if (source.hasAttribute("END") && builder.getAlleles().stream().noneMatch(Allele::isSymbolic)) {
            builder.attribute("END", (int)builder.getStop());
        } else {
            builder.rmAttribute("END");
        }
        builder.rmAttribute(SWAPPED_ALLELES);
        if (target.isNegativeStrand()) {
            builder.attribute(REV_COMPED_ALLELES, true);
        } else {
            builder.rmAttribute(REV_COMPED_ALLELES);
        }
        builder.id(source.getID());
        if (writeOriginalPosition) {
            builder.attribute("OriginalContig", source.getContig());
            builder.attribute("OriginalStart", source.getStart());
        }
        if (writeOriginalAlleles && !source.getAlleles().equals(builder.getAlleles())) {
            builder.attribute("OriginalAlleles", LiftoverUtils.allelesToStringList(source.getAlleles()));
        }
        return builder.make();
    }

    protected static List<String> allelesToStringList(List<Allele> alleles) {
        ArrayList<String> ret = new ArrayList<String>();
        alleles.forEach(a -> ret.add(a.isNoCall() ? "." : a.getDisplayString()));
        return ret;
    }

    protected static VariantContextBuilder liftSimpleVariantContext(VariantContext source, Interval target) {
        if (target == null || source.getReference().length() != target.length()) {
            return null;
        }
        VariantContextBuilder builder = new VariantContextBuilder(source);
        builder.chr(target.getContig());
        builder.start(target.getStart());
        builder.stop(target.getEnd());
        return builder;
    }

    protected static VariantContextBuilder reverseComplementVariantContext(VariantContext source, Interval target, ReferenceSequence refSeq) {
        if (target.isPositiveStrand()) {
            throw new IllegalArgumentException("This should only be called for negative strand liftovers");
        }
        int start = target.getStart();
        int stop = target.getEnd();
        ArrayList<Allele> origAlleles = new ArrayList<Allele>(source.getAlleles());
        VariantContextBuilder vcb = new VariantContextBuilder(source);
        vcb.rmAttribute("END").chr(target.getContig()).start(start).stop(stop).alleles((Collection<Allele>)LiftoverUtils.reverseComplementAlleles(origAlleles));
        if (LiftoverUtils.isIndelForLiftover(source)) {
            if (LiftoverUtils.referenceAlleleDiffersFromReferenceForIndel(vcb.getAlleles(), refSeq, start, stop)) {
                return null;
            }
            LiftoverUtils.leftAlignVariant(vcb, start, stop, vcb.getAlleles(), refSeq);
        }
        vcb.genotypes(LiftoverUtils.fixGenotypes(source.getGenotypes(), origAlleles, vcb.getAlleles()));
        return vcb;
    }

    private static boolean isIndelForLiftover(VariantContext vc) {
        Allele ref = vc.getReference();
        if (ref.length() != 1) {
            return vc.getAlternateAlleles().stream().anyMatch(a -> !a.isSymbolic() && !a.equals(Allele.SPAN_DEL));
        }
        return vc.getAlleles().stream().filter(a -> !a.isSymbolic()).filter(a -> !a.equals(Allele.SPAN_DEL)).anyMatch(a -> a.length() != 1);
    }

    private static List<Allele> reverseComplementAlleles(List<Allele> originalAlleles) {
        return originalAlleles.stream().map(LiftoverUtils::reverseComplement).collect(Collectors.toList());
    }

    private static Allele reverseComplement(Allele oldAllele) {
        if (oldAllele.isSymbolic() || oldAllele.isNoCall() || oldAllele.equals(Allele.SPAN_DEL)) {
            return oldAllele;
        }
        return Allele.create(SequenceUtil.reverseComplement(oldAllele.getBaseString()), oldAllele.isReference());
    }

    protected static GenotypesContext fixGenotypes(GenotypesContext originals, List<Allele> originalAlleles, List<Allele> newAlleles) {
        if (originalAlleles.equals(newAlleles)) {
            return originals;
        }
        if (originalAlleles.size() != newAlleles.size()) {
            throw new IllegalStateException("Error in allele lists: the original and new allele lists are not the same length: " + originalAlleles.toString() + " / " + newAlleles.toString());
        }
        HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>();
        for (int idx = 0; idx < originalAlleles.size(); ++idx) {
            alleleMap.put(originalAlleles.get(idx), newAlleles.get(idx));
        }
        GenotypesContext fixedGenotypes = GenotypesContext.create(originals.size());
        for (Genotype genotype : originals) {
            ArrayList<Allele> fixedAlleles = new ArrayList<Allele>();
            for (Allele allele : genotype.getAlleles()) {
                if (allele.isNoCall()) {
                    fixedAlleles.add(allele);
                    continue;
                }
                Allele newAllele = (Allele)alleleMap.get(allele);
                if (newAllele == null) {
                    throw new IllegalStateException("Allele not found: " + allele.toString() + ", " + String.valueOf(originalAlleles) + "/ " + String.valueOf(newAlleles));
                }
                fixedAlleles.add(newAllele);
            }
            fixedGenotypes.add(new GenotypeBuilder(genotype).alleles(fixedAlleles).make());
        }
        return fixedGenotypes;
    }

    public static VariantContext swapRefAlt(VariantContext vc, Collection<String> annotationsToReverse, Collection<String> annotationsToDrop) {
        if (!vc.isBiallelic() || !vc.isSNP()) {
            throw new IllegalArgumentException("swapRefAlt can only process biallelic, SNPS, found " + vc.toString());
        }
        VariantContextBuilder swappedBuilder = new VariantContextBuilder(vc);
        swappedBuilder.attribute(SWAPPED_ALLELES, true);
        swappedBuilder.alleles(Arrays.asList(vc.getAlleles().get(1).getBaseString(), vc.getAlleles().get(0).getBaseString()));
        HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>();
        alleleMap.put(vc.getAlleles().get(0), swappedBuilder.getAlleles().get(1));
        alleleMap.put(vc.getAlleles().get(1), swappedBuilder.getAlleles().get(0));
        GenotypesContext swappedGenotypes = GenotypesContext.create(vc.getGenotypes().size());
        for (Genotype genotype : vc.getGenotypes()) {
            ArrayList<Allele> swappedAlleles = new ArrayList<Allele>();
            for (Allele allele : genotype.getAlleles()) {
                if (allele.isNoCall()) {
                    swappedAlleles.add(allele);
                    continue;
                }
                swappedAlleles.add((Allele)alleleMap.get(allele));
            }
            GenotypeBuilder builder = new GenotypeBuilder(genotype).alleles(swappedAlleles);
            if (genotype.hasAD() && genotype.getAD().length == 2) {
                int[] ad = ArrayUtils.clone(genotype.getAD());
                ArrayUtils.reverse(ad);
                builder.AD(ad);
            } else {
                builder.noAD();
            }
            if (genotype.hasPL() && genotype.getPL().length == 3) {
                int[] pl = ArrayUtils.clone(genotype.getPL());
                ArrayUtils.reverse(pl);
                builder.PL(pl);
            } else {
                builder.noPL();
            }
            swappedGenotypes.add(builder.make());
        }
        swappedBuilder.genotypes(swappedGenotypes);
        for (String key : vc.getAttributes().keySet()) {
            if (annotationsToDrop.contains(key)) {
                swappedBuilder.rmAttribute(key);
                continue;
            }
            if (!annotationsToReverse.contains(key) || vc.getAttributeAsString(key, "").equals(".")) continue;
            double attributeToReverse = vc.getAttributeAsDouble(key, -1.0);
            if (attributeToReverse < 0.0 || attributeToReverse > 1.0) {
                log.warn("Trying to reverse attribute " + key + " but found value that isn't between 0 and 1: (" + attributeToReverse + ") in variant " + String.valueOf(vc) + ". Results might be wrong.");
            }
            swappedBuilder.attribute(key, 1.0 - attributeToReverse);
        }
        return swappedBuilder.make();
    }

    private static boolean referenceAlleleDiffersFromReferenceForIndel(List<Allele> alleles, ReferenceSequence referenceSequence, int start, int end) {
        Allele refAllele;
        String refString = StringUtil.bytesToString(referenceSequence.getBases(), start - 1, end - start + 1);
        return !refString.equalsIgnoreCase((refAllele = alleles.stream().filter(Allele::isReference).findAny().orElseThrow(() -> new IllegalStateException("Error: no reference allele was present"))).getBaseString());
    }

    protected static void leftAlignVariant(VariantContextBuilder builder, int start, int end, List<Allele> alleles, ReferenceSequence referenceSequence) {
        int oldEnd;
        int oldStart;
        if (LiftoverUtils.referenceAlleleDiffersFromReferenceForIndel(alleles, referenceSequence, start, end)) {
            throw new IllegalArgumentException(String.format("Reference allele doesn't match reference at %s:%d-%d", referenceSequence.getName(), start, end));
        }
        HashMap<Allele, byte[]> alleleBasesMap = new HashMap<Allele, byte[]>();
        alleles.stream().filter(a -> !a.equals(Allele.SPAN_DEL) && !a.isSymbolic()).forEach(a -> alleleBasesMap.put((Allele)a, a.getBases()));
        int theStart = start;
        int theEnd = end;
        do {
            boolean extendLeft;
            byte extraBase;
            oldStart = theStart;
            oldEnd = theEnd;
            if (alleleBasesMap.values().stream().collect(Collectors.groupingBy(a -> a[((byte[])a).length - 1], Collectors.toSet())).size() == 1 && theEnd > 1) {
                alleleBasesMap.replaceAll((a, v) -> LiftoverUtils.truncateBase((byte[])alleleBasesMap.get(a), true));
                --theEnd;
            }
            if (!alleleBasesMap.values().stream().map(a -> ((byte[])a).length).anyMatch(l -> l == 0)) continue;
            if (theStart > 1) {
                extraBase = referenceSequence.getBases()[theStart - 2];
                extendLeft = true;
                --theStart;
            } else {
                extraBase = referenceSequence.getBases()[theEnd];
                extendLeft = false;
                ++theEnd;
            }
            for (Allele allele : alleleBasesMap.keySet()) {
                alleleBasesMap.put(allele, LiftoverUtils.extendOneBase((byte[])alleleBasesMap.get(allele), extraBase, extendLeft));
            }
        } while (theStart != oldStart || theEnd != oldEnd);
        while (alleleBasesMap.values().stream().allMatch(a -> ((byte[])a).length >= 2) && alleleBasesMap.values().stream().collect(Collectors.groupingBy(a -> a[0], Collectors.toSet())).size() == 1) {
            alleleBasesMap.replaceAll((a, v) -> LiftoverUtils.truncateBase((byte[])alleleBasesMap.get(a), false));
            ++theStart;
        }
        builder.start(theStart);
        builder.stop(theEnd);
        Map<Allele, Allele> fixedAlleleMap = alleleBasesMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, me -> Allele.create((byte[])me.getValue(), ((Allele)me.getKey()).isReference())));
        fixedAlleleMap.put(Allele.SPAN_DEL, Allele.SPAN_DEL);
        alleles.stream().filter(Allele::isSymbolic).forEach(a -> fixedAlleleMap.put((Allele)a, (Allele)a));
        List<Allele> fixedAlleles = alleles.stream().map(fixedAlleleMap::get).collect(Collectors.toList());
        builder.alleles((Collection<Allele>)fixedAlleles);
    }

    private static byte[] truncateBase(byte[] allele, boolean truncateRightmost) {
        return Arrays.copyOfRange(allele, truncateRightmost ? 0 : 1, truncateRightmost ? allele.length - 1 : allele.length);
    }

    private static byte[] extendOneBase(byte[] bases, byte base) {
        return LiftoverUtils.extendOneBase(bases, base, true);
    }

    private static byte[] extendOneBase(byte[] bases, byte base, boolean extendLeft) {
        byte[] newBases = new byte[bases.length + 1];
        System.arraycopy(bases, 0, newBases, extendLeft ? 1 : 0, bases.length);
        newBases[extendLeft ? 0 : bases.length] = base;
        return newBases;
    }
}

