/*
 * Decompiled with CFR 0.152.
 */
package picard.vcf;

import htsjdk.io.HtsPath;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.seekablestream.SeekablePathStream;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;
import htsjdk.samtools.util.BlockCompressedStreamConstants;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.FileExtensions;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFContigHeaderLine;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLine;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.VariantManipulationProgramGroup;

@CommandLineProgramProperties(summary="Gathers multiple VCF files from a scatter operation into a single VCF file. Input files must be supplied in genomic order and must not have events at overlapping positions.", oneLineSummary="Gathers multiple VCF files from a scatter operation into a single VCF file", programGroup=VariantManipulationProgramGroup.class)
@DocumentedFeature
public class GatherVcfs
extends CommandLineProgram {
    @Argument(shortName="I", doc="Input VCF file(s).")
    public List<HtsPath> INPUT;
    @Argument(shortName="O", doc="Output VCF file.")
    public File OUTPUT;
    @Argument(doc="Comment(s) to include in the merged output file's header.", optional=true, shortName="CO")
    public List<String> COMMENT = new ArrayList<String>();
    @Argument(doc="If 'true' the program will reorder INPUT according to the genomic location of the first variant in each file. this is useful since the order of variants in each file in INPUT come from non overlapping regions  but the order of the files in INPUT is untrusted.", optional=true, shortName="RI")
    public boolean REORDER_INPUT_BY_FIRST_VARIANT = false;
    private static final Log log = Log.getInstance(GatherVcfs.class);

    public GatherVcfs() {
        this.CREATE_INDEX = true;
    }

    @Override
    protected int doWork() {
        log.info("Checking inputs.");
        List<Path> paths = this.INPUT.stream().map(HtsPath::toPath).collect(Collectors.toList());
        List<Path> unrolledPaths = IOUtil.unrollPaths(paths, FileExtensions.VCF_LIST.toArray(new String[0]));
        IOUtil.assertPathsAreReadable(unrolledPaths);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        SAMSequenceDictionary sequenceDictionary = VCFFileReader.getSequenceDictionary(unrolledPaths.get(0));
        if (this.CREATE_INDEX.booleanValue() && sequenceDictionary == null) {
            throw new PicardException("In order to index the resulting VCF input VCFs must contain ##contig lines.");
        }
        log.info("Checking file headers and first records to ensure compatibility.");
        try {
            unrolledPaths = this.assertSameSamplesAndValidOrdering(unrolledPaths);
            if (this.areAllBlockCompressed(unrolledPaths) && this.areAllBlockCompressed(Collections.singletonList(this.OUTPUT.toPath()))) {
                log.info("Gathering by copying gzip blocks. Will not be able to validate position non-overlap of files.");
                if (this.CREATE_INDEX.booleanValue()) {
                    log.warn("Index creation not currently supported when gathering block compressed VCFs.");
                }
                GatherVcfs.gatherWithBlockCopying(unrolledPaths, this.OUTPUT);
            } else {
                log.info("Gathering by conventional means.");
                GatherVcfs.gatherConventionally(sequenceDictionary, this.CREATE_INDEX, unrolledPaths, this.OUTPUT, this.COMMENT);
            }
        }
        catch (RuntimeException e) {
            log.error("There was a problem with gathering the INPUT.", e);
            try {
                Files.deleteIfExists(this.OUTPUT.toPath());
            }
            catch (Exception exception) {
                // empty catch block
            }
            return 1;
        }
        return 0;
    }

    private boolean areAllBlockCompressed(List<Path> input) {
        for (Path path : input) {
            if (!VCFFileReader.isBCF(path) && IOUtil.hasBlockCompressedExtension(path)) continue;
            return false;
        }
        return true;
    }

    private List<Path> assertSameSamplesAndValidOrdering(List<Path> inputFiles) {
        VCFHeader header;
        try (VCFFileReader reader = new VCFFileReader(inputFiles.get(0), false);){
            header = reader.getFileHeader();
        }
        SAMSequenceDictionary dict = header.getSequenceDictionary();
        VariantContextComparator comparator = new VariantContextComparator(header.getSequenceDictionary());
        List<String> samples = header.getGenotypeSamples();
        Path lastFile = null;
        VariantContext lastContext = null;
        if (this.REORDER_INPUT_BY_FIRST_VARIANT) {
            ArrayList<FirstVariantInVcf> filesandvariants = new ArrayList<FirstVariantInVcf>(inputFiles.size());
            for (Path path : inputFiles) {
                FirstVariantInVcf vcfcxt = new FirstVariantInVcf(path);
                try (VCFFileReader in = new VCFFileReader(path, false);
                     Iterator iter = in.iterator();){
                    VariantContext variantContext = vcfcxt.firstVariant = iter.hasNext() ? (VariantContext)iter.next() : null;
                    if (vcfcxt.firstVariant == null) {
                        log.info("No variant in " + String.valueOf(path));
                    }
                }
                filesandvariants.add(vcfcxt);
            }
            filesandvariants.sort((A2, B) -> {
                if (A2.firstVariant == null) {
                    if (B.firstVariant == null) {
                        return 0;
                    }
                    return 1;
                }
                if (B.firstVariant == null) {
                    return -1;
                }
                return comparator.compare(A2.firstVariant, B.firstVariant);
            });
            inputFiles.clear();
            inputFiles.addAll(filesandvariants.stream().map(FV -> FV.vcfFile).collect(Collectors.toList()));
        }
        for (Path path : inputFiles) {
            VCFFileReader in = new VCFFileReader(path, false);
            try {
                dict.assertSameDictionary(in.getFileHeader().getSequenceDictionary());
            }
            catch (AssertionError e) {
                log.error("File #1: " + String.valueOf(inputFiles.get(0)));
                log.error("File #2: " + String.valueOf(path));
                throw e;
            }
            List<String> theseSamples = in.getFileHeader().getGenotypeSamples();
            if (!samples.equals(theseSamples)) {
                TreeSet<String> s1 = new TreeSet<String>(samples);
                TreeSet<String> s2 = new TreeSet<String>(theseSamples);
                s1.removeAll(theseSamples);
                s2.removeAll(samples);
                throw new IllegalArgumentException("VCFs do not have identical sample lists. Samples unique to first file: " + String.valueOf(s1) + ". Samples unique to " + String.valueOf(path.toAbsolutePath()) + ": " + String.valueOf(s2) + ".");
            }
            Iterator variantIterator = in.iterator();
            if (variantIterator.hasNext()) {
                VariantContext currentContext = (VariantContext)variantIterator.next();
                if (lastContext != null && comparator.compare(lastContext, currentContext) >= 0) {
                    throw new IllegalArgumentException("First record in file " + String.valueOf(path.toAbsolutePath()) + " is not after first record in previous file " + String.valueOf(lastFile.toAbsolutePath()));
                }
                lastContext = currentContext;
                lastFile = path;
            }
            CloserUtil.close(in);
        }
        return inputFiles;
    }

    private static void gatherConventionally(SAMSequenceDictionary sequenceDictionary, boolean createIndex, List<Path> inputFiles, File outputFile, List<String> comments) {
        EnumSet<Options> options = EnumSet.copyOf(VariantContextWriterBuilder.DEFAULT_OPTIONS);
        if (createIndex) {
            options.add(Options.INDEX_ON_THE_FLY);
        } else {
            options.remove((Object)Options.INDEX_ON_THE_FLY);
        }
        VariantContextWriter out = new VariantContextWriterBuilder().setOptions(options).setOutputFile(outputFile).setReferenceDictionary(sequenceDictionary).build();
        ProgressLogger progress = new ProgressLogger(log, 10000);
        VariantContext lastContext = null;
        Path lastFile = null;
        VCFHeader firstHeader = null;
        VariantContextComparator comparator = null;
        for (Path path : inputFiles) {
            VariantContext vc;
            log.debug("Gathering from file: ", path.toAbsolutePath());
            VCFFileReader variantReader = new VCFFileReader(path, false);
            PeekableIterator variantIterator = new PeekableIterator(variantReader.iterator());
            VCFHeader header = variantReader.getFileHeader();
            if (firstHeader == null) {
                firstHeader = header;
                for (String comment : comments) {
                    firstHeader.addMetaDataLine(new VCFHeaderLine("GatherVcfs.comment", comment));
                }
                out.writeHeader(firstHeader);
                comparator = new VariantContextComparator((Collection<VCFContigHeaderLine>)firstHeader.getContigLines());
            }
            if (lastContext != null && variantIterator.hasNext() && comparator.compare(vc = (VariantContext)variantIterator.peek(), lastContext) <= 0) {
                throw new IllegalArgumentException("First variant in file " + String.valueOf(path.toAbsolutePath()) + " is at " + vc.getContig() + ":" + vc.getStart() + " but last variant in earlier file " + String.valueOf(lastFile.toAbsolutePath()) + " is at " + lastContext.getContig() + ":" + lastContext.getStart());
            }
            while (variantIterator.hasNext()) {
                lastContext = (VariantContext)variantIterator.next();
                out.add(lastContext);
                progress.record(lastContext.getContig(), lastContext.getStart());
            }
            lastFile = path;
            CloserUtil.close(variantIterator);
            CloserUtil.close(variantReader);
        }
        out.close();
    }

    private static void gatherWithBlockCopying(List<Path> vcfs, File output) {
        try {
            FileOutputStream out = new FileOutputStream(output);
            boolean isFirstFile = true;
            for (Path path : vcfs) {
                log.info("Gathering " + String.valueOf(path.toAbsolutePath()));
                SeekablePathStream seekableStream = new SeekablePathStream(path);
                BlockCompressedInputStream.FileTermination term = BlockCompressedInputStream.checkTermination(path);
                if (term == BlockCompressedInputStream.FileTermination.DEFECTIVE) {
                    throw new PicardException(String.valueOf(path.toAbsolutePath()) + " does not have a valid GZIP block at the end of the file.");
                }
                if (!isFirstFile) {
                    BlockCompressedInputStream blockIn = new BlockCompressedInputStream((InputStream)seekableStream, false);
                    boolean lastByteNewline = true;
                    while (blockIn.available() > 0) {
                        int blockLength = blockIn.available();
                        byte[] blockContents = new byte[blockLength];
                        int read = blockIn.read(blockContents);
                        if (blockLength == 0 || read != blockLength) {
                            throw new IllegalStateException("Could not read available bytes from BlockCompressedInputStream.");
                        }
                        int firstNonHeaderByteIndex = -1;
                        for (int i = 0; i < read; ++i) {
                            boolean thisByteNewline;
                            byte b = blockContents[i];
                            boolean bl = thisByteNewline = b == 10 || b == 13;
                            if (lastByteNewline && !thisByteNewline && b != 35) {
                                firstNonHeaderByteIndex = i;
                                break;
                            }
                            lastByteNewline = thisByteNewline;
                        }
                        if (firstNonHeaderByteIndex < 0) continue;
                        BlockCompressedOutputStream blockOut = new BlockCompressedOutputStream((OutputStream)out, (Path)null);
                        blockOut.write(blockContents, firstNonHeaderByteIndex, blockContents.length - firstNonHeaderByteIndex);
                        blockOut.flush();
                        break;
                    }
                }
                long currentPos = ((SeekableStream)seekableStream).position();
                long length = Files.size(path);
                long skipLast = term == BlockCompressedInputStream.FileTermination.HAS_TERMINATOR_BLOCK ? (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length : 0L;
                long bytesToWrite = length - skipLast - currentPos;
                IOUtil.transferByStream(seekableStream, out, bytesToWrite);
                ((SeekableStream)seekableStream).close();
                isFirstFile = false;
            }
            out.write(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK);
            out.close();
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe);
        }
    }

    private static class FirstVariantInVcf {
        final Path vcfFile;
        VariantContext firstVariant = null;

        FirstVariantInVcf(Path vcfFile) {
            this.vcfFile = vcfFile;
        }
    }
}

