/*
 * Decompiled with CFR 0.152.
 */
package picard.illumina;

import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProcessExecutor;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.BaseCallingProgramGroup;
import picard.illumina.BasecallsConverter;
import picard.illumina.IlluminaBasecallsConverter;
import picard.illumina.NewIlluminaBasecallsConverter;
import picard.illumina.parser.IlluminaDataProviderFactory;
import picard.illumina.parser.IlluminaDataType;
import picard.illumina.parser.IlluminaFileUtil;
import picard.illumina.parser.NewIlluminaDataProvider;
import picard.illumina.parser.OutputMapping;
import picard.illumina.parser.ParameterizedFileUtil;
import picard.illumina.parser.ReadStructure;
import picard.illumina.parser.readers.AbstractIlluminaPositionFileReader;
import picard.illumina.parser.readers.BaseBclReader;
import picard.illumina.parser.readers.CbclReader;
import picard.illumina.parser.readers.LocsFileReader;

@CommandLineProgramProperties(summary="Asserts the validity for specified Illumina basecalling data.  <p>This tool will check that the basecall directory and the internal files are available, exist, and are reasonably sized for every tile and cycle.  Reasonably sized means non-zero sized for files that exist per tile and equal size for binary files that exist per cycle or per tile. If DATA_TYPES {Position, BaseCalls, QualityScores, PF, or Barcodes} are not specified, then the default data types used by IlluminaBasecallsToSam are used.  CheckIlluminaDirectory DOES NOT check that the individual records in a file are well-formed. If there are errors, the number of errors is written in a file called 'errors.count' in the working directory</p><h4>Usage example:</h4> <pre>java -jar picard.jar CheckIlluminaDirectory \\<br />      BASECALLS_DIR=/BaseCalls/  \\<br />      READ_STRUCTURE=25T8B25T \\<br />      LANES=1 \\<br />      DATA_TYPES=BaseCalls </pre><hr />", oneLineSummary="Asserts the validity for specified Illumina basecalling data.  ", programGroup=BaseCallingProgramGroup.class)
@DocumentedFeature
public class CheckIlluminaDirectory
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Asserts the validity for specified Illumina basecalling data.  ";
    static final String USAGE_DETAILS = "<p>This tool will check that the basecall directory and the internal files are available, exist, and are reasonably sized for every tile and cycle.  Reasonably sized means non-zero sized for files that exist per tile and equal size for binary files that exist per cycle or per tile. If DATA_TYPES {Position, BaseCalls, QualityScores, PF, or Barcodes} are not specified, then the default data types used by IlluminaBasecallsToSam are used.  CheckIlluminaDirectory DOES NOT check that the individual records in a file are well-formed. If there are errors, the number of errors is written in a file called 'errors.count' in the working directory</p><h4>Usage example:</h4> <pre>java -jar picard.jar CheckIlluminaDirectory \\<br />      BASECALLS_DIR=/BaseCalls/  \\<br />      READ_STRUCTURE=25T8B25T \\<br />      LANES=1 \\<br />      DATA_TYPES=BaseCalls </pre><hr />";
    private static final Log log = Log.getInstance(CheckIlluminaDirectory.class);
    @Argument(doc="The basecalls output directory. ", shortName="B")
    public File BASECALLS_DIR;
    @Argument(doc="The data types that should be checked for each tile/cycle.  If no values are provided then the data types checked are those required by IlluminaBaseCallsToSam (which is a superset of those used in ExtractIlluminaBarcodes).  These data types vary slightly depending on whether or not the run is barcoded so READ_STRUCTURE should be the same as that which will be passed to IlluminaBasecallsToSam.  If this option is left unspecified then both ExtractIlluminaBarcodes and IlluminaBaseCallsToSam should complete successfully UNLESS the individual records of the files themselves are spurious.", shortName="DT", optional=true)
    public Set<IlluminaDataType> DATA_TYPES = new TreeSet<IlluminaDataType>();
    @Argument(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Sample Barcode, M for molecular barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"28T8M8B8S28T\" then the sequence may be split up into four reads:\n* read one with 28 cycles (bases) of template\n* read two with 8 cycles (bases) of molecular barcode (ex. unique molecular barcode)\n* read three with 8 cycles (bases) of sample barcode\n* 8 cycles (bases) skipped.\n* read four with 28 cycles (bases) of template\nThe skipped cycles would NOT be included in an output SAM/BAM file or in read groups therein. Note:  If you want to check whether or not a future IlluminaBasecallsToSam or ExtractIlluminaBarcodes run will fail then be sure to use the exact same READ_STRUCTURE that you would pass to these programs for this run.", shortName="RS")
    public String READ_STRUCTURE;
    @Argument(doc="The number of the lane(s) to check. ", shortName="L", minElements=1)
    public List<Integer> LANES;
    @Argument(doc="The number(s) of the tile(s) to check. ", shortName="T", optional=true)
    public List<Integer> TILE_NUMBERS;
    @Argument(doc="A flag to determine whether or not to create fake versions of the missing files.", shortName="F", optional=true)
    public Boolean FAKE_FILES = false;
    @Deprecated
    @Argument(doc="A flag to create symlinks to the loc file for the X Ten for each tile. @deprecated It is no longer necessary to create locs file symlinks.", shortName="X", optional=true)
    public Boolean LINK_LOCS = false;

    public static void main(String[] argv) {
        new CheckIlluminaDirectory().instanceMainWithExit(argv);
    }

    @Override
    protected int doWork() {
        ReadStructure readStructure = new ReadStructure(this.READ_STRUCTURE);
        if (this.DATA_TYPES.isEmpty()) {
            this.DATA_TYPES.addAll(Arrays.asList(IlluminaBasecallsConverter.DATA_TYPES_NO_BARCODE));
        }
        ArrayList<Integer> failingLanes = new ArrayList<Integer>();
        int totalFailures = 0;
        OutputMapping outputMapping = new OutputMapping(readStructure);
        log.info("Checking lanes(" + StringUtil.join(",", this.LANES) + " in basecalls directory (" + this.BASECALLS_DIR.getAbsolutePath() + ")\n");
        log.info("Expected cycles: " + StringUtil.intValuesToString(outputMapping.getOutputCycles()));
        for (Integer lane : this.LANES) {
            if (IlluminaFileUtil.hasCbcls(this.BASECALLS_DIR, lane)) {
                File[] filterFiles;
                ArrayList<Integer> tiles = new ArrayList<Integer>();
                File laneDir = new File(this.BASECALLS_DIR, IlluminaFileUtil.longLaneStr(lane));
                File[] cycleDirs = IOUtil.getFilesMatchingRegexp(laneDir, IlluminaFileUtil.CYCLE_SUBDIRECTORY_PATTERN);
                ArrayList<File> cbcls = new ArrayList<File>();
                Arrays.asList(cycleDirs).forEach(cycleDir -> cbcls.addAll(Arrays.asList(IOUtil.getFilesMatchingRegexp(cycleDir, "^" + IlluminaFileUtil.longLaneStr(lane) + "_(\\d{1,5}).cbcl$"))));
                IOUtil.assertFilesAreReadable(cbcls);
                Pattern laneTileRegex = Pattern.compile(ParameterizedFileUtil.escapePeriods(ParameterizedFileUtil.makeLaneTileRegex(".filter", lane)));
                for (File filterFile : filterFiles = NewIlluminaBasecallsConverter.getTiledFiles(laneDir, laneTileRegex)) {
                    Matcher tileMatcher = laneTileRegex.matcher(filterFile.getName());
                    if (!tileMatcher.matches()) continue;
                    tiles.add(Integer.valueOf(tileMatcher.group(1)));
                }
                IOUtil.assertFilesAreReadable(Arrays.asList(filterFiles));
                tiles.sort(BasecallsConverter.TILE_NUMBER_COMPARATOR);
                File locsFile = new File(this.BASECALLS_DIR.getParentFile(), "s.locs");
                ArrayList<AbstractIlluminaPositionFileReader.PositionInfo> locs = new ArrayList<AbstractIlluminaPositionFileReader.PositionInfo>();
                HashMap<Integer, File> filterFileMap = new HashMap<Integer, File>();
                try (File[] locsFileReader = new LocsFileReader(locsFile);){
                    while (locsFileReader.hasNext()) {
                        locs.add(locsFileReader.next());
                    }
                }
                for (File filterFile : filterFiles) {
                    filterFileMap.put(NewIlluminaDataProvider.fileToTile(filterFile.getName()), filterFile);
                }
                CbclReader reader = new CbclReader(cbcls, filterFileMap, outputMapping.getOutputReadLengths(), (Integer)tiles.get(0), locs, outputMapping.getOutputCycles(), true);
                var17_22 = null;
                try {
                    reader.getAllTiles().forEach((key, value) -> {
                        String emptyCycleString = value.stream().filter(cycle -> cycle.getCompressedBlockSize() <= 2).map(BaseBclReader.TileData::getTileNum).map(Object::toString).collect(Collectors.joining(", "));
                        if (emptyCycleString.length() > 0) {
                            log.warn("The following tiles have no data for cycle " + key);
                            log.warn(emptyCycleString);
                        }
                        List<File> fileForCycle = reader.getFilesForCycle((int)key);
                        long totalFilesSize = fileForCycle.stream().mapToLong(file -> file.length() - (long)reader.getHeaderSize()).sum();
                        long expectedFileSize = value.stream().mapToLong(BaseBclReader.TileData::getCompressedBlockSize).sum();
                        if (expectedFileSize != totalFilesSize) {
                            throw new PicardException(String.format("File %s is not the expected size of %d instead it is %d", fileForCycle, expectedFileSize, totalFilesSize));
                        }
                    });
                    continue;
                }
                catch (Throwable throwable) {
                    var17_22 = throwable;
                    throw throwable;
                }
                finally {
                    if (reader == null) continue;
                    if (var17_22 != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable) {
                            var17_22.addSuppressed(throwable);
                        }
                        continue;
                    }
                    reader.close();
                    continue;
                }
            }
            IlluminaFileUtil fileUtil = new IlluminaFileUtil(this.BASECALLS_DIR, lane);
            List<Integer> expectedTiles = fileUtil.getExpectedTiles();
            if (!this.TILE_NUMBERS.isEmpty()) {
                expectedTiles.retainAll(this.TILE_NUMBERS);
            }
            if (this.LINK_LOCS.booleanValue()) {
                this.createLocFileSymlinks(fileUtil, lane);
                fileUtil = new IlluminaFileUtil(this.BASECALLS_DIR, lane);
            }
            log.info("Checking lane " + lane);
            log.info("Expected tiles: " + StringUtil.join(", ", expectedTiles));
            int numFailures = CheckIlluminaDirectory.verifyLane(fileUtil, expectedTiles, outputMapping.getOutputCycles(), this.DATA_TYPES, this.FAKE_FILES);
            if (numFailures > 0) {
                log.info("Lane " + lane + " FAILED  Total Errors: " + numFailures);
                failingLanes.add(lane);
                totalFailures += numFailures;
                continue;
            }
            log.info("Lane " + lane + " SUCCEEDED ");
        }
        int status = 0;
        if (totalFailures == 0) {
            log.info("SUCCEEDED!  All required files are present and non-empty.");
        } else {
            status = 1;
            try {
                Files.write(Paths.get("./errors.count", new String[0]), Integer.toString(totalFailures).getBytes(), new OpenOption[0]);
            }
            catch (IOException e) {
                log.error("Unable to write number of errors to file", e);
            }
            log.info("FAILED! There were " + totalFailures + " in the following lanes: " + StringUtil.join(", ", failingLanes));
        }
        return status;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void createLocFileSymlinks(IlluminaFileUtil fileUtil, int lane) {
        File baseFile = new File(this.BASECALLS_DIR.getParentFile().getAbsolutePath() + File.separator + "s.locs");
        File newFileBase = new File(baseFile.getParent() + File.separator + IlluminaFileUtil.longLaneStr(lane) + File.separator);
        if (!baseFile.exists()) throw new PicardException(String.format("Locations file %s does not exist.", baseFile.getAbsolutePath()));
        boolean success = true;
        if (!newFileBase.exists()) {
            success = newFileBase.mkdirs();
        }
        if (!success) throw new PicardException(String.format("Could not create lane directory: %s.", newFileBase.getAbsolutePath()));
        for (Integer tile : fileUtil.getExpectedTiles()) {
            String newName = newFileBase + File.separator + String.format("s_%d_%d.locs", lane, tile);
            ProcessExecutor.ExitStatusAndOutput output = ProcessExecutor.executeAndReturnInterleavedOutput(new String[]{"ln", "-fs", baseFile.getAbsolutePath(), newName});
            if (output.exitStatus == 0) continue;
            throw new PicardException("Could not create symlink: " + output.stdout);
        }
    }

    private static int verifyLane(IlluminaFileUtil fileUtil, List<Integer> expectedTiles, int[] cycles, Set<IlluminaDataType> dataTypes, boolean fakeFiles) {
        if (expectedTiles.isEmpty()) {
            throw new PicardException("0 input tiles were specified!  Check to make sure this lane is in the InterOp file!");
        }
        if (cycles.length == 0) {
            throw new PicardException("0 output cycles were specified!");
        }
        int numFailures = 0;
        Map<IlluminaFileUtil.SupportedIlluminaFormat, Set<IlluminaDataType>> formatToDataTypes = IlluminaDataProviderFactory.determineFormats(dataTypes, fileUtil);
        Set<IlluminaDataType> unmatchedDataTypes = IlluminaDataProviderFactory.findUnmatchedTypes(dataTypes, formatToDataTypes);
        if (!unmatchedDataTypes.isEmpty()) {
            if (fakeFiles) {
                for (IlluminaDataType dataType : unmatchedDataTypes) {
                    IlluminaFileUtil.SupportedIlluminaFormat format = IlluminaDataProviderFactory.findPreferredFormat(dataType, fileUtil);
                    fileUtil.getUtil(format).fakeFiles(expectedTiles, cycles, format);
                }
            }
            log.info("Could not find a format with available files for the following data types: " + StringUtil.join(", ", new ArrayList<IlluminaDataType>(unmatchedDataTypes)));
            numFailures += unmatchedDataTypes.size();
        }
        for (IlluminaFileUtil.SupportedIlluminaFormat format : formatToDataTypes.keySet()) {
            ParameterizedFileUtil util = fileUtil.getUtil(format);
            util.setTilesForPerRunFile(expectedTiles);
            List<String> failures = util.verify(expectedTiles, cycles);
            if (!failures.isEmpty() && fakeFiles) {
                util.fakeFiles(expectedTiles, cycles, format);
            }
            numFailures += failures.size();
            for (String failure : failures) {
                log.info(failure);
            }
        }
        return numFailures;
    }

    @Override
    protected String[] customCommandLineValidation() {
        IOUtil.assertDirectoryIsReadable(this.BASECALLS_DIR);
        ArrayList<String> errors = new ArrayList<String>();
        for (Integer lane : this.LANES) {
            if (lane >= 1) continue;
            errors.add("LANES must be greater than or equal to 1.  LANES passed in " + StringUtil.join(", ", this.LANES));
            break;
        }
        if (errors.isEmpty()) {
            return null;
        }
        return errors.toArray(new String[errors.size()]);
    }
}

