/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.http.nio;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.List;
import java.util.Map;
import org.broadinstitute.http.nio.HttpFileSystemProviderSettings;
import org.broadinstitute.http.nio.IncompatibleResponseToRangeQueryException;
import org.broadinstitute.http.nio.RetryHandler;
import org.broadinstitute.http.nio.UnexpectedHttpResponseException;
import org.broadinstitute.http.nio.utils.HttpUtils;
import org.broadinstitute.http.nio.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpSeekableByteChannel
implements SeekableByteChannel {
    private static final long SKIP_DISTANCE = 8192L;
    private static final Logger LOGGER = LoggerFactory.getLogger(HttpSeekableByteChannel.class);
    private final URI uri;
    private final RetryHandler retryHandler;
    private final HttpClient client;
    private ReadableByteChannel channel = null;
    private InputStream backingStream = null;
    private long position = 0L;
    private long size = -1L;

    public HttpSeekableByteChannel(URI uri) throws IOException {
        this(uri, HttpFileSystemProviderSettings.DEFAULT_SETTINGS, 0L);
    }

    public HttpSeekableByteChannel(URI uri, long position) throws IOException {
        this(uri, HttpFileSystemProviderSettings.DEFAULT_SETTINGS, position);
    }

    public HttpSeekableByteChannel(URI uri, HttpFileSystemProviderSettings settings, long position) throws IOException {
        this.uri = Utils.nonNull(uri, () -> "null URI");
        this.client = HttpUtils.getClient(Utils.nonNull(settings, () -> "settings"));
        this.retryHandler = new RetryHandler(settings.retrySettings(), uri);
        this.retryHandler.runWithRetries(() -> this.openChannel(position));
    }

    @Override
    public synchronized int read(ByteBuffer dst) throws IOException {
        this.assertChannelIsOpen();
        int read = this.retryHandler.tryOnceThenWithRetries(() -> HttpSeekableByteChannel.readWithoutPerturbingTheBufferIfAnErrorOccurs(dst, this.channel), () -> {
            this.closeSilently();
            this.openChannel(this.position);
            return HttpSeekableByteChannel.readWithoutPerturbingTheBufferIfAnErrorOccurs(dst, this.channel);
        });
        if (read != -1) {
            this.position += (long)read;
        }
        return read;
    }

    public static int readWithoutPerturbingTheBufferIfAnErrorOccurs(ByteBuffer dst, ReadableByteChannel channel) throws IOException {
        ByteBuffer copy = dst.duplicate();
        copy.order(dst.order());
        int read = channel.read(copy);
        dst.position(copy.position());
        return read;
    }

    private void assertChannelIsOpen() throws ClosedChannelException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }

    @Override
    public int write(ByteBuffer src) {
        throw new NonWritableChannelException();
    }

    @Override
    public synchronized long position() throws IOException {
        this.assertChannelIsOpen();
        return this.position;
    }

    @Override
    public synchronized HttpSeekableByteChannel position(long newPosition) throws IOException {
        this.assertChannelIsOpen();
        Utils.validateArg(newPosition >= 0L, "Cannot seek to a negative position (from " + this.position + " to " + newPosition + " ).");
        if (this.position == newPosition) {
            return this;
        }
        if (this.position < newPosition && newPosition - this.position < 8192L) {
            this.retryHandler.tryOnceThenWithRetries(() -> {
                long bytesToSkip = newPosition - this.position;
                this.backingStream.skipNBytes(bytesToSkip);
                LOGGER.debug("Skipped {} bytes out of {} when setting position to {} (previously on {})", bytesToSkip, bytesToSkip, newPosition, this.position);
                return null;
            }, () -> {
                this.closeSilently();
                this.openChannel(newPosition);
                return null;
            });
        } else {
            this.closeSilently();
            this.retryHandler.runWithRetries(() -> this.openChannel(newPosition));
        }
        this.position = newPosition;
        return this;
    }

    @Override
    public synchronized long size() throws IOException {
        this.assertChannelIsOpen();
        this.retryHandler.runWithRetries(() -> {
            if (this.size == -1L) {
                HttpRequest headRequest = HttpRequest.newBuilder().uri(this.uri).method("HEAD", HttpRequest.BodyPublishers.noBody()).build();
                try {
                    HttpResponse<Void> response = this.client.send(headRequest, HttpResponse.BodyHandlers.discarding());
                    this.assertGoodHttpResponse(response, false);
                    Map<String, List<String>> map = response.headers().map();
                    List<String> contentLengthStrings = map.get("content-length");
                    if (contentLengthStrings == null || contentLengthStrings.size() != 1) {
                        throw new IOException("Failed to get size of file at " + this.uri.toString() + ", content-length=" + contentLengthStrings);
                    }
                    this.size = Long.parseLong(contentLengthStrings.get(0));
                }
                catch (InterruptedException e) {
                    throw new InterruptedIOException("Interrupted while trying to get size of file at " + this.uri.toString());
                }
            }
        });
        return this.size;
    }

    private void assertGoodHttpResponse(HttpResponse<?> response, boolean isRangeRequest) throws FileNotFoundException, UnexpectedHttpResponseException {
        int code = response.statusCode();
        switch (code) {
            case 200: {
                if (!isRangeRequest) break;
                throw new IncompatibleResponseToRangeQueryException(200, "Server returned entire file instead of subrange for " + this.uri);
            }
            case 206: {
                if (isRangeRequest) break;
                throw new IncompatibleResponseToRangeQueryException(206, "Unexpected Partial Content result for request for entire file at " + this.uri);
            }
            case 404: {
                throw new FileNotFoundException("File not found at " + this.uri + " got http 404 response.");
            }
            default: {
                throw new UnexpectedHttpResponseException(code, "Unexpected http response code: " + code + " when requesting " + this.uri);
            }
        }
    }

    @Override
    public SeekableByteChannel truncate(long size) {
        throw new NonWritableChannelException();
    }

    @Override
    public synchronized boolean isOpen() {
        return this.channel.isOpen();
    }

    @Override
    public synchronized void close() throws IOException {
        this.channel.close();
    }

    private synchronized void closeSilently() {
        try {
            this.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private synchronized void openChannel(long position) throws IOException {
        HttpResponse<InputStream> response;
        boolean isRangeRequest;
        HttpRequest.Builder builder = HttpRequest.newBuilder(this.uri).GET();
        boolean bl = isRangeRequest = position != 0L;
        if (isRangeRequest) {
            builder.setHeader("Range", "bytes=" + position + "-");
        }
        HttpRequest request = builder.build();
        try {
            response = this.client.send(request, HttpResponse.BodyHandlers.ofInputStream());
        }
        catch (FileNotFoundException ex) {
            throw ex;
        }
        catch (IOException ex) {
            throw new IOException("Failed to connect to " + this.uri + " at position: " + position, ex);
        }
        catch (InterruptedException ex) {
            throw new InterruptedIOException("Interrupted while connecting to " + this.uri + " at position: " + position);
        }
        this.assertGoodHttpResponse(response, isRangeRequest);
        this.backingStream = new BufferedInputStream(response.body());
        this.channel = Channels.newChannel(this.backingStream);
        this.position = position;
    }
}

