/*
 * Decompiled with CFR 0.152.
 */
package picard.illumina;

import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.xml.parsers.DocumentBuilderFactory;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.BaseCallingProgramGroup;
import picard.illumina.IlluminaLaneMetrics;
import picard.illumina.IlluminaPhasingMetrics;
import picard.illumina.parser.ReadDescriptor;
import picard.illumina.parser.ReadStructure;
import picard.illumina.parser.ReadType;
import picard.illumina.parser.Tile;
import picard.illumina.parser.TileMetricsUtil;
import picard.illumina.parser.readers.TileMetricsOutReader;

@CommandLineProgramProperties(summary="Collects Illumina lane metrics for the given BaseCalling analysis directory.This tool produces quality control metrics on cluster density for each lane of an Illumina flowcell. This tool takes Illumina TileMetrics data and places them into directories containing lane- and phasing-level metrics. In this context, phasing refers to the fraction of molecules that fall behind or jump ahead (prephasing) during a read cycle.<h4>Usage example:</h4><pre>java -jar picard.jar CollectIlluminaLaneMetrics \\<br />      RUN_DIR=test_run \\<br />      OUTPUT_DIRECTORY=Lane_output_metrics \\<br />      OUTPUT_PREFIX=experiment1 \\<br />      READ_STRUCTURE=25T8B25T </pre><p>Please see the CollectIlluminaLaneMetrics <a href='http://broadinstitute.github.io/picard/picard-metric-definitions.html#CollectIlluminaLaneMetrics'>definitions</a> for a complete description of the metrics produced by this tool.</p><hr />", oneLineSummary="Collects Illumina lane metrics for the given BaseCalling analysis directory.", programGroup=BaseCallingProgramGroup.class)
@DocumentedFeature
public class CollectIlluminaLaneMetrics
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Collects Illumina lane metrics for the given BaseCalling analysis directory.";
    static final String USAGE_DETAILS = "This tool produces quality control metrics on cluster density for each lane of an Illumina flowcell. This tool takes Illumina TileMetrics data and places them into directories containing lane- and phasing-level metrics. In this context, phasing refers to the fraction of molecules that fall behind or jump ahead (prephasing) during a read cycle.<h4>Usage example:</h4><pre>java -jar picard.jar CollectIlluminaLaneMetrics \\<br />      RUN_DIR=test_run \\<br />      OUTPUT_DIRECTORY=Lane_output_metrics \\<br />      OUTPUT_PREFIX=experiment1 \\<br />      READ_STRUCTURE=25T8B25T </pre><p>Please see the CollectIlluminaLaneMetrics <a href='http://broadinstitute.github.io/picard/picard-metric-definitions.html#CollectIlluminaLaneMetrics'>definitions</a> for a complete description of the metrics produced by this tool.</p><hr />";
    @Argument(doc="The Illumina run directory of the run for which the lane metrics are to be generated")
    public File RUN_DIRECTORY;
    @Argument(doc="The directory to which the output file will be written")
    public File OUTPUT_DIRECTORY;
    @Argument(doc="The prefix to be prepended to the file name of the output file; an appropriate suffix will be applied", shortName="O")
    public String OUTPUT_PREFIX;
    @Argument(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Sample Barcode, M for molecular barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"28T8M8B8S28T\" then the sequence may be split up into four reads:\n* read one with 28 cycles (bases) of template\n* read two with 8 cycles (bases) of molecular barcode (ex. unique molecular barcode)\n* read three with 8 cycles (bases) of sample barcode\n* 8 cycles (bases) skipped.\n* read four with 28 cycles (bases) of template\nThe skipped cycles would NOT be included in an output SAM/BAM file or in read groups therein.\nIf not given, will use the RunInfo.xml in the run directory.", shortName="RS", optional=true)
    public ReadStructure READ_STRUCTURE;
    @Argument(doc="Append the given file extension to all metric file names (ex. OUTPUT.illumina_lane_metrics.EXT). None if null", shortName="EXT", optional=true)
    public String FILE_EXTENSION = null;

    @Override
    protected int doWork() {
        MetricsFile<MetricBase, Comparable<?>> laneMetricsFile = this.getMetricsFile();
        MetricsFile<MetricBase, Comparable<?>> phasingMetricsFile = this.getMetricsFile();
        if (this.READ_STRUCTURE == null) {
            File runInfo = new File(String.valueOf(this.RUN_DIRECTORY) + "/RunInfo.xml");
            IOUtil.assertFileIsReadable(runInfo);
            try {
                Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(runInfo);
                NodeList reads = document.getElementsByTagName("Read");
                ArrayList<ReadDescriptor> descriptors = new ArrayList<ReadDescriptor>(reads.getLength());
                for (int i = 0; i < reads.getLength(); ++i) {
                    Node read = reads.item(i);
                    NamedNodeMap attributes = read.getAttributes();
                    int readNumber = Integer.parseInt(attributes.getNamedItem("Number").getNodeValue());
                    int numCycles = Integer.parseInt(attributes.getNamedItem("NumCycles").getNodeValue());
                    boolean isIndexedRead = attributes.getNamedItem("IsIndexedRead").getNodeValue().toUpperCase().equals("Y");
                    if (readNumber != i + 1) {
                        throw new PicardException("Read number in RunInfo.xml was out of order: " + (i + 1) + " != " + readNumber);
                    }
                    descriptors.add(new ReadDescriptor(numCycles, isIndexedRead ? ReadType.Barcode : ReadType.Template));
                }
                this.READ_STRUCTURE = new ReadStructure(descriptors);
            }
            catch (Exception e) {
                throw new PicardException(e.getMessage());
            }
        }
        IlluminaLaneMetricsCollector.collectLaneMetrics(this.RUN_DIRECTORY, this.OUTPUT_DIRECTORY, this.OUTPUT_PREFIX, laneMetricsFile, phasingMetricsFile, this.READ_STRUCTURE, this.FILE_EXTENSION == null ? "" : this.FILE_EXTENSION, this.VALIDATION_STRINGENCY);
        return 0;
    }

    public static class IlluminaLaneMetricsCollector {
        private static final Log LOG = Log.getInstance(IlluminaLaneMetricsCollector.class);

        public static Map<Integer, ? extends Collection<Tile>> readLaneTiles(File illuminaRunDirectory, ReadStructure readStructure, ValidationStringency validationStringency, int tileMetricsVersion) {
            List<File> tileMetricsOutFiles = TileMetricsUtil.findTileMetricsFiles(illuminaRunDirectory, readStructure.totalCycles);
            Collection<Tile> tiles = tileMetricsVersion == TileMetricsOutReader.TileMetricsVersion.THREE.version ? TileMetricsUtil.parseClusterRecordsFromTileMetrics(tileMetricsOutFiles, TileMetricsUtil.renderPhasingMetricsFilesFromBasecallingDirectory(illuminaRunDirectory), readStructure) : TileMetricsUtil.parseTileMetrics(tileMetricsOutFiles.get(0), readStructure, validationStringency);
            return tiles.stream().filter(tile -> tile.getLaneNumber() > 0).collect(Collectors.groupingBy(Tile::getLaneNumber));
        }

        public static void collectLaneMetrics(File runDirectory, File outputDirectory, String outputPrefix, MetricsFile<MetricBase, Comparable<?>> laneMetricsFile, MetricsFile<MetricBase, Comparable<?>> phasingMetricsFile, ReadStructure readStructure, String fileExtension, ValidationStringency validationStringency) {
            int tileMetricsVersion = IlluminaLaneMetricsCollector.determineTileMetricsVersion(runDirectory, readStructure);
            Map<Integer, ? extends Collection<Tile>> laneTiles = IlluminaLaneMetricsCollector.readLaneTiles(runDirectory, readStructure, validationStringency, tileMetricsVersion);
            IlluminaLaneMetricsCollector.writeLaneMetrics(laneTiles, outputDirectory, outputPrefix, laneMetricsFile, fileExtension);
            IlluminaLaneMetricsCollector.writePhasingMetrics(laneTiles, outputDirectory, outputPrefix, phasingMetricsFile, fileExtension, tileMetricsVersion);
        }

        private static int determineTileMetricsVersion(File illuminaRunDirectory, ReadStructure readStructure) {
            List<File> tileMetricsOutFiles = TileMetricsUtil.findTileMetricsFiles(illuminaRunDirectory, readStructure.totalCycles);
            int version = new TileMetricsOutReader(tileMetricsOutFiles.get(0)).getVersion();
            if (!tileMetricsOutFiles.stream().allMatch(metricFile -> {
                boolean matches;
                int fileVersion = new TileMetricsOutReader((File)metricFile).getVersion();
                boolean bl = matches = fileVersion == version;
                if (!matches) {
                    LOG.error(String.format("File %s version %d does not match expected version %d.", metricFile.getAbsolutePath(), fileVersion, version));
                }
                return matches;
            })) {
                throw new PicardException("Not all tile metrics files match expected version: " + version);
            }
            return version;
        }

        private static void writePhasingMetrics(Map<Integer, ? extends Collection<Tile>> laneTiles, File outputDirectory, String outputPrefix, MetricsFile<MetricBase, Comparable<?>> phasingMetricsFile, String fileExtension, int tileMetricsVersion) {
            laneTiles.forEach((key, value) -> IlluminaPhasingMetrics.getPhasingMetricsForTiles(key.longValue(), value, tileMetricsVersion == TileMetricsOutReader.TileMetricsVersion.TWO.version).forEach(phasingMetricsFile::addMetric));
            IlluminaLaneMetricsCollector.writeMetrics(phasingMetricsFile, outputDirectory, outputPrefix, IlluminaPhasingMetrics.getExtension() + fileExtension);
        }

        private static void writeLaneMetrics(Map<Integer, ? extends Collection<Tile>> laneTiles, File outputDirectory, String outputPrefix, MetricsFile<MetricBase, Comparable<?>> laneMetricsFile, String fileExtension) {
            laneTiles.forEach((key, value) -> {
                IlluminaLaneMetrics laneMetric = new IlluminaLaneMetrics();
                laneMetric.LANE = key.longValue();
                laneMetric.CLUSTER_DENSITY = IlluminaLaneMetricsCollector.calculateLaneDensityFromTiles(value);
                laneMetricsFile.addMetric(laneMetric);
            });
            IlluminaLaneMetricsCollector.writeMetrics(laneMetricsFile, outputDirectory, outputPrefix, IlluminaLaneMetrics.getExtension() + fileExtension);
        }

        private static void writeMetrics(MetricsFile<MetricBase, Comparable<?>> metricsFile, File outputDirectory, String outputPrefix, String outputExtension) {
            File outputFile = new File(outputDirectory, String.format("%s.%s", outputPrefix, outputExtension));
            LOG.info(String.format("Writing %s lane metrics to %s ...", metricsFile.getMetrics().size(), outputFile));
            metricsFile.write(outputFile);
        }

        private static double calculateLaneDensityFromTiles(Collection<Tile> tiles) {
            double area = 0.0;
            double clusters = 0.0;
            for (Tile tile : tiles) {
                if (tile.getClusterDensity() > 0.0f) {
                    area += (double)(tile.getClusterCount() / tile.getClusterDensity());
                }
                clusters += (double)tile.getClusterCount();
            }
            return area > 0.0 ? clusters / area : 0.0;
        }
    }
}

