/*
 * Decompiled with CFR 0.152.
 */
package picard.util;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.tribble.AbstractFeatureReader;
import htsjdk.tribble.CloseableTribbleIterator;
import htsjdk.tribble.annotation.Strand;
import htsjdk.tribble.bed.BEDCodec;
import htsjdk.tribble.bed.BEDFeature;
import htsjdk.tribble.readers.LineIterator;
import htsjdk.variant.utils.SAMSequenceDictionaryExtractor;
import java.io.File;
import java.io.IOException;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.argparser.Hidden;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.IntervalsManipulationProgramGroup;

@CommandLineProgramProperties(summary="Converts a BED file to a Picard Interval List.  This tool provides easy conversion from BED to the Picard interval_list format which is required by many Picard processing tools. Note that the coordinate system of BED files is such that the first base or position in a sequence is numbered \"0\", while in interval_list files it is numbered \"1\".<br /><br />BED files contain sequence data displayed in a flexible format that includes nine optional fields, in addition to three required fields within the annotation tracks. The required fields of a BED file include:<pre>     chrom - The name of the chromosome (e.g. chr20) or scaffold (e.g. scaffold10671) <br />     chromStart - The starting position of the feature in the chromosome or scaffold. The first base in a chromosome is numbered \"0\" <br />     chromEnd - The ending position of the feature in the chromosome or scaffold.  The chromEnd base is not included in the display of the feature. For example, the first 100 bases of a chromosome are defined as chromStart=0, chromEnd=100, and span the bases numbered 0-99.</pre>In each annotation track, the number of fields per line must be consistent throughout a data set. For additional information regarding BED files and the annotation field options, please see: http://genome.ucsc.edu/FAQ/FAQformat.html#format1.<br /> <br /> Interval_list files contain sequence data distributed into intervals. The interval_list file format is relatively simple and reflects the SAM alignment format to a degree.  A SAM style header must be present in the file that lists the sequence records against which the intervals are described.  After the header, the file then contains records, one per line in plain text format with the following values tab-separated::<pre>      -Sequence name (SN) - The name of the sequence in the file for identification purposes, can be chromosome number e.g. chr20 <br />      -Start position - Interval start position (starts at +1) <br />      -End position - Interval end position (1-based, end inclusive) <br />      -Strand - Indicates +/- strand for the interval (either + or -) <br />      -Interval name - (Each interval should have a unique name) </pre><br/>This tool requires a sequence dictionary, provided with the SEQUENCE_DICTIONARY or SD argument. The value given to this argument can be any of the following:<pre>    - A file with .dict extension generated using Picard's CreateSequenceDictionaryTool</br>    - A reference.fa or reference.fasta file with a reference.dict in the same directory</br>    - Another IntervalList with @SQ lines in the header from which to generate a dictionary</br>    - A VCF that contains #contig lines from which to generate a sequence dictionary</br>    - A SAM or BAM file with @SQ lines in the header from which to generate a dictionary</br></pre><h4>Usage example:</h4><pre>java -jar picard.jar BedToIntervalList \\<br />      I=input.bed \\<br />      O=list.interval_list \\<br />      SD=reference_sequence.dict</pre><br /> <br /> <hr />", oneLineSummary="Converts a BED file to a Picard Interval List.  ", programGroup=IntervalsManipulationProgramGroup.class)
@DocumentedFeature
public class BedToIntervalList
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Converts a BED file to a Picard Interval List.  ";
    static final String USAGE_DETAILS = "This tool provides easy conversion from BED to the Picard interval_list format which is required by many Picard processing tools. Note that the coordinate system of BED files is such that the first base or position in a sequence is numbered \"0\", while in interval_list files it is numbered \"1\".<br /><br />BED files contain sequence data displayed in a flexible format that includes nine optional fields, in addition to three required fields within the annotation tracks. The required fields of a BED file include:<pre>     chrom - The name of the chromosome (e.g. chr20) or scaffold (e.g. scaffold10671) <br />     chromStart - The starting position of the feature in the chromosome or scaffold. The first base in a chromosome is numbered \"0\" <br />     chromEnd - The ending position of the feature in the chromosome or scaffold.  The chromEnd base is not included in the display of the feature. For example, the first 100 bases of a chromosome are defined as chromStart=0, chromEnd=100, and span the bases numbered 0-99.</pre>In each annotation track, the number of fields per line must be consistent throughout a data set. For additional information regarding BED files and the annotation field options, please see: http://genome.ucsc.edu/FAQ/FAQformat.html#format1.<br /> <br /> Interval_list files contain sequence data distributed into intervals. The interval_list file format is relatively simple and reflects the SAM alignment format to a degree.  A SAM style header must be present in the file that lists the sequence records against which the intervals are described.  After the header, the file then contains records, one per line in plain text format with the following values tab-separated::<pre>      -Sequence name (SN) - The name of the sequence in the file for identification purposes, can be chromosome number e.g. chr20 <br />      -Start position - Interval start position (starts at +1) <br />      -End position - Interval end position (1-based, end inclusive) <br />      -Strand - Indicates +/- strand for the interval (either + or -) <br />      -Interval name - (Each interval should have a unique name) </pre><br/>This tool requires a sequence dictionary, provided with the SEQUENCE_DICTIONARY or SD argument. The value given to this argument can be any of the following:<pre>    - A file with .dict extension generated using Picard's CreateSequenceDictionaryTool</br>    - A reference.fa or reference.fasta file with a reference.dict in the same directory</br>    - Another IntervalList with @SQ lines in the header from which to generate a dictionary</br>    - A VCF that contains #contig lines from which to generate a sequence dictionary</br>    - A SAM or BAM file with @SQ lines in the header from which to generate a dictionary</br></pre><h4>Usage example:</h4><pre>java -jar picard.jar BedToIntervalList \\<br />      I=input.bed \\<br />      O=list.interval_list \\<br />      SD=reference_sequence.dict</pre><br /> <br /> <hr />";
    @Argument(shortName="I", doc="The input BED file")
    public File INPUT;
    @Argument(shortName="SD", doc="The sequence dictionary, or BAM/VCF/IntervalList from which a dictionary can be extracted.")
    public File SEQUENCE_DICTIONARY;
    @Argument(shortName="O", doc="The output Picard Interval List")
    public File OUTPUT;
    @Argument(doc="If true, sort the output interval list before writing it.")
    public boolean SORT = true;
    @Argument(doc="If true, unique the output interval list by merging overlapping regions, before writing it (implies sort=true).")
    public boolean UNIQUE = false;
    @Hidden
    @Argument(doc="If true, entries that are on contig-names that are missing from the provided dictionary will be dropped.")
    public boolean DROP_MISSING_CONTIGS = false;
    @Argument(doc="If true, write length zero intervals in input bed file to resulting interval list file.")
    public boolean KEEP_LENGTH_ZERO_INTERVALS = false;
    private final Log LOG = Log.getInstance(this.getClass());
    private int missingIntervals = 0;
    private int missingRegion = 0;
    private int lengthZeroIntervals = 0;

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.INPUT);
        if (this.INPUT.getPath().equals("/dev/stdin")) {
            throw new IllegalArgumentException("BedToIntervalList does not support reading from standard input - a file must be provided.");
        }
        IOUtil.assertFileIsReadable(this.SEQUENCE_DICTIONARY);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        try {
            SAMFileHeader header = new SAMFileHeader();
            SAMSequenceDictionary samSequenceDictionary = SAMSequenceDictionaryExtractor.extractDictionary(this.SEQUENCE_DICTIONARY.toPath());
            header.setSequenceDictionary(samSequenceDictionary);
            header.setSortOrder(SAMFileHeader.SortOrder.coordinate);
            IntervalList intervalList = new IntervalList(header);
            AbstractFeatureReader<BEDFeature, LineIterator> bedReader = AbstractFeatureReader.getFeatureReader(this.INPUT.getAbsolutePath(), new BEDCodec(), false);
            CloseableTribbleIterator iterator = bedReader.iterator();
            ProgressLogger progressLogger = new ProgressLogger(this.LOG, 1000000);
            while (iterator.hasNext()) {
                boolean isNegativeStrand;
                BEDFeature bedFeature = (BEDFeature)iterator.next();
                String sequenceName = bedFeature.getContig();
                int start = bedFeature.getStart();
                int end = bedFeature.getEnd();
                String name = bedFeature.getName().isEmpty() ? null : bedFeature.getName();
                SAMSequenceRecord sequenceRecord = header.getSequenceDictionary().getSequence(sequenceName);
                if (null == sequenceRecord) {
                    if (this.DROP_MISSING_CONTIGS) {
                        this.LOG.info(String.format("Dropping interval with missing contig: %s:%d-%d", sequenceName, bedFeature.getStart(), bedFeature.getEnd()));
                        ++this.missingIntervals;
                        this.missingRegion += bedFeature.getEnd() - bedFeature.getStart();
                        continue;
                    }
                    throw new PicardException(String.format("Sequence '%s' was not found in the sequence dictionary", sequenceName));
                }
                if (start < 1) {
                    throw new PicardException(String.format("Start on sequence '%s' was less than one: %d", sequenceName, start));
                }
                if (sequenceRecord.getSequenceLength() < start) {
                    throw new PicardException(String.format("Start on sequence '%s' was past the end: %d < %d", sequenceName, sequenceRecord.getSequenceLength(), start));
                }
                if (end == 0 && start != 1 || end < 0) {
                    throw new PicardException(String.format("End on sequence '%s' was less than one: %d", sequenceName, end));
                }
                if (sequenceRecord.getSequenceLength() < end) {
                    throw new PicardException(String.format("End on sequence '%s' was past the end: %d < %d", sequenceName, sequenceRecord.getSequenceLength(), end));
                }
                if (end < start - 1) {
                    throw new PicardException(String.format("On sequence '%s', end < start-1: %d <= %d", sequenceName, end, start));
                }
                boolean bl = isNegativeStrand = bedFeature.getStrand() == Strand.NEGATIVE;
                if (start == end + 1 && !this.KEEP_LENGTH_ZERO_INTERVALS) {
                    this.LOG.info(String.format("Skipping writing length zero interval at %s:%d-%d.", sequenceName, start, end));
                } else {
                    Interval interval = new Interval(sequenceName, start, end, isNegativeStrand, name);
                    intervalList.add(interval);
                }
                if (start == end + 1) {
                    ++this.lengthZeroIntervals;
                }
                progressLogger.record(sequenceName, start);
            }
            CloserUtil.close(bedReader);
            if (this.DROP_MISSING_CONTIGS) {
                if (this.missingRegion == 0) {
                    this.LOG.info("There were no missing regions.");
                } else {
                    this.LOG.warn(String.format("There were %d missing regions with a total of %d bases", this.missingIntervals, this.missingRegion));
                }
            }
            if (!this.KEEP_LENGTH_ZERO_INTERVALS) {
                if (this.lengthZeroIntervals == 0) {
                    this.LOG.info("No input regions had length zero, so none were skipped.");
                } else {
                    this.LOG.info(String.format("Skipped writing a total of %d entries with length zero in the input file.", this.lengthZeroIntervals));
                }
            } else if (this.lengthZeroIntervals > 0) {
                this.LOG.warn(String.format("Input file had %d entries with length zero. Run with the KEEP_LENGTH_ZERO_INTERVALS flag set to false to remove these.", this.lengthZeroIntervals));
            }
            IntervalList out = intervalList;
            if (this.SORT) {
                out = out.sorted();
            }
            if (this.UNIQUE) {
                out = out.uniqued();
            }
            out.write(this.OUTPUT);
            this.LOG.info(String.format("Wrote %d intervals spanning a total of %d bases", out.getIntervals().size(), out.getBaseCount()));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return 0;
    }
}

